import fs from 'fs/promises'

export default function buildIndexFile() {
    let dir, bundleName, distHtml, input, srcHtml, distBlade

    async function contentReplace(
        file,
        pattern,
        replace,
        write = true,
        debug = false
    ) {
        const content = await fs.readFile(file, {
            encoding: 'utf-8',
        })

        const updated = content.replace(pattern, replace)

        const _debug = debug
        if (_debug) {
            console.trace()
            console.log(file)
            console.log('content')
            console.log(content)
            console.log('updated')
            console.log(updated)
        }

        if (write) await fs.writeFile(file, updated)
    }

    async function injectBundleScriptTag() {
        await contentReplace(
            distHtml,
            /<script.*id="dev".*>/,
            `<script async src="{{ asset('${bundleName}') }}"></script>`
        )
    }

    async function clearNewLines() {
        await contentReplace(distHtml, /\n+/g, '\n')
    }

    async function stripHtmlComments() {
        await contentReplace(distBlade, /<!--(.|\n)*-->/g, '')
    }

    async function copyIndexToOutputDir() {
        await fs.copyFile('index.html', distHtml)
    }

    async function unCommentProductionImports() {
        await contentReplace(input, /\/\/production\s(.*)/g, (match, $1) => {
            return `${$1}//production comment`
        })
    }

    async function commentProductionImports() {
        await contentReplace(
            input,
            /(.*)\/\/production comment/g,
            (match, $1) => {
                return `//production ${$1}`
            }
        )
    }

    async function commentNormalize() {
        await contentReplace(
            srcHtml,
            /<link\s+id="normalize"(.|\n)*?\/>/g,
            (match) => {
                return `<!--${match}-->`
            }
        )
    }

    async function unCommentNormalize() {
        await contentReplace(
            srcHtml,
            /<!--(<link\s+id="normalize"(.|\n)*?\/>)-->/,
            (match, $1) => {
                return $1
            }
        )
    }

    async function bindBladeConfig() {
        await contentReplace(
            distHtml,
            /'config.(.*?)':(.|\n)*?'(.*?)'/g,
            (match, $1) => {
                const result = `'config.${$1}': '{!! config( "${$1}" ) !!}'`

                return result
            }
        )

        await fs.rename(distHtml, distBlade)
    }

    async function injectBladeComment() {
        const content = await fs.readFile(distBlade, { encoding: 'utf-8' })

        const comment = `{{-- 
This is auto generated file. 
Do not change content manually. 
Any changes will be discarded after the next build. 
--}}`

        await fs.writeFile(distBlade, comment + '\n' + content)
    }

    async function injectBladeDirectives() {
        await contentReplace(
            distBlade,
            /<!-- blade-directive: (.*) -->/g,
            ($0, $1) => {
                return $1
            }
        )
    }

    async function bindPageTitle() {
        await contentReplace(
            distBlade,
            /<title>QR Code Generator<\/title>/,
            `<title>
            @section('page-title')
                {{ PageTitle::makeTitle() }}
            @show
        </title>`
        )
    }

    async function transformAssetsUrls() {
        await contentReplace(distBlade, /"\/(assets\/.*)"/g, ($0, $1) => {
            return `{{ asset('${$1}') }}`
        })
    }

    async function removeFromBladeCommand() {
        await contentReplace(
            distBlade,
            /<!-- remove-from-blade -->(.|\n)*?<!-- \/remove-from-blade -->/g,
            ''
        )
    }

    return {
        name: 'build-index-file', // this name will show up in warnings and errors

        buildStart: async function (options) {
            input = options.input[0]

            srcHtml = 'index.html'

            await unCommentProductionImports()

            await commentNormalize()
        },

        writeBundle: async function (options, bundle) {
            bundleName = Object.keys(bundle)[0]

            dir = options.dir

            distHtml = dir + '/index.html'

            distBlade = dir + '/index.blade.php'

            await copyIndexToOutputDir()

            await injectBundleScriptTag()

            await clearNewLines()

            await commentProductionImports()

            await unCommentNormalize()

            await bindBladeConfig()

            await injectBladeComment()

            await injectBladeDirectives()

            await transformAssetsUrls()

            await removeFromBladeCommand()

            await stripHtmlComments()

            await bindPageTitle()
        },
    }
}
