import { LitElement, html, css } from 'lit'

import '../ui/qrcg-box'
import '../ui/qrcg-input'
import '../ui/qrcg-button'
import '../ui/qrcg-form'
import '../ui/qrcg-link'
import { post } from '../core/api'
import { QRCGApiConsumer } from '../core/qrcg-api-consumer'
import { QRCGFormController } from '../core/qrcg-form-controller'
import { styled } from '../core/helpers'
import { t } from '../core/translate'
import { Config } from '../core/qrcg-config'

export class QRCGAccountPage extends LitElement {
    api = new QRCGApiConsumer(this)

    formController = new QRCGFormController(this)

    static get styles() {
        return css`
            :host {
                display: block;
                max-width: 25rem;
                margin: auto;
                margin-top: 3rem;
            }

            qrcg-form::part(form) {
                display: grid;
                grid-gap: 1rem;
            }

            .go-home {
                display: block;
                color: var(--primary-0);
                margin-top: 2rem;
                text-align: center;
            }

            h1 {
                border-bottom: 2px solid var(--gray-1);
                padding-bottom: 1rem;
                margin-top: 1rem;
                font-size: 1.3rem;
            }

            p {
                line-height: 1.8;
            }
        `
    }

    static get properties() {
        return {
            data: {},
        }
    }

    connectedCallback() {
        super.connectedCallback()

        this.attachGlobalStyles()
    }

    disconnectedCallback() {
        super.disconnectedCallback()

        this.detachGlobalStyles()
    }

    attachGlobalStyles() {
        this.globalStyles = document.createElement('style')

        this.globalStyles.innerHTML = styled`
            html {
                font-size: calc(14px + 6 * ((100vw - 320px) / 1240));
            }

            @media (min-width: 800px) {
                html {
                    font-size: calc(12px + 6 * ((100vw - 320px) / 1240));
                }
            }
            

            @media (min-width: 2000px) {
                html {
                    font-size: calc(12px + 6 * ((200vw) / 1240));
                }
            }
        `

        document.head.appendChild(this.globalStyles)
    }

    detachGlobalStyles() {
        this.globalStyles.remove()
    }

    async submitForm() {
        try {
            const { token, user } = await this.api.call(() =>
                post('login', this.data)
            )

            window.dispatchEvent(
                new CustomEvent('qrcg-login:after-login', {
                    detail: {
                        user,
                        token,
                    },
                })
            )
        } catch (error) {
            console.error(error)
        }
    }

    renderForm() {}

    renderTitle() {}

    renderContent() {
        return html`
            ${this.renderTitle()}

            <qrcg-form> ${this.renderForm()} </qrcg-form>
        `
    }

    shouldRenderGoHomeLink() {
        const config = Config.get('app.frontend_links')

        return !config || config === 'enabled'
    }

    render() {
        return html`
            <qrcg-box> ${this.renderContent()} </qrcg-box>

            ${this.shouldRenderGoHomeLink()
                ? html`
                      <a class="go-home" href="/"
                          >${t`Or you may go back to the home page.`}</a
                      >
                  `
                : null}
        `
    }
}
