import { html, css } from 'lit'

import '../ui/qrcg-box'
import '../ui/qrcg-input'
import '../ui/qrcg-button'
import '../ui/qrcg-form'
import '../ui/qrcg-link'
import { post } from '../core/api'

import { QRCGAccountPage } from './qrcg-account-page'
import { Config } from '../core/qrcg-config'
import { push } from '../core/qrcg-router'
import { t } from '../core/translate'

export class QRCGLogin extends QRCGAccountPage {
    static get styles() {
        return [
            super.styles,
            css`
                qrcg-form::part(form) {
                    grid-template-areas:
                        'email email'
                        'password password'
                        'reset signup'
                        'submit submit';
                }
                [name='email'] {
                    grid-area: email;
                }
                [name='password'] {
                    grid-area: password;
                }
                .signup {
                    text-align: right;
                }
                qrcg-button {
                    grid-area: submit;
                }
            `,
        ]
    }

    static get properties() {
        return {
            data: {},
            installed: { type: Boolean },
        }
    }

    constructor() {
        super()

        this.data = {
            email: '',
            password: '',
        }
    }

    connectedCallback() {
        super.connectedCallback()

        this.installed = new URLSearchParams(window.location.search).get(
            'installed'
        )

        if (this.installed === 'true') {
            return
        }

        if (Config.get('app.env') === 'local') {
            this.data = {
                email: 'mohammad.a.alhomsi@gmail.com',
                password: 'password',
            }
        } else if (Config.get('app.env') === 'demo') {
            this.data = {
                email: 'admin@example.invalid',
                password: 'password',
            }
        }
    }

    async submitForm() {
        const login = this.constructor.login.bind(this)

        return login(this.data)
    }

    static async login(data) {
        try {
            const apiCall = async (cp) => {
                if (this.api) {
                    return this.api.call(cp)
                }

                const { response } = await cp()

                const json = await response.clone().json()

                return json
            }

            const { token, user } = await apiCall(() => post('login', data))

            if (this.installed) {
                this.addRedirectToSystemStatusPage()
            }

            window.dispatchEvent(
                new CustomEvent('qrcg-login:after-login', {
                    detail: {
                        user,
                        token,
                    },
                })
            )
        } catch (error) {
            console.error(error)
        }
    }

    renderTitle() {
        return html`<h1>${t`Login`}</h1>`
    }

    addRedirectToSystemStatusPage() {
        const searchParams = new URLSearchParams(location.search)

        searchParams.set('redirect', '/dashboard/system/status')

        push(location.pathname + '?' + searchParams.toString())
    }

    renderForm() {
        return html`
            <qrcg-input
                autofocus
                name="email"
                placeholder=${t`email@provider.com`}
            >
                ${t`Email`}
            </qrcg-input>

            <qrcg-input
                name="password"
                type="password"
                placeholder=${t`your password`}
            >
                ${t`Password`}
            </qrcg-input>

            <qrcg-link href="/account/forgot-password"
                >${t`Did you forget your password?`}</qrcg-link
            >

            ${Config.get('app.new_user_registration') === 'disabled'
                ? null
                : html`
                      <qrcg-link href="/account/sign-up" class="signup">
                          ${t`Register`}
                      </qrcg-link>
                  `}

            <qrcg-button type="submit">${t`Login`}</qrcg-button>
        `
    }
}

window.customElements.define('qrcg-login', QRCGLogin)
