import { html, css } from 'lit'
import { userHomePage, validateCurrentToken } from '../core/auth'
import { queryParam } from '../core/helpers'
import { QRCGApiConsumer } from '../core/qrcg-api-consumer'
import { push } from '../core/qrcg-router'
import { t } from '../core/translate'
import { showToast } from '../ui/qrcg-toast'
import { QrcgCheckoutPage } from './qrcg-checkout-page'

export class QrcgCheckoutStripeSuccess extends QrcgCheckoutPage {
    api = QRCGApiConsumer.instance({
        host: this,
        baseRoute: 'checkout/stripe/verify-checkout-session',
        bindEvents: true,
        loadableElementsSelector: '.loader-container',
    })

    static styles = [
        super.styles,
        css`
            :host {
                display: block;
            }

            .loader-container:not([loading]) {
                display: none;
            }
        `,
    ]

    static get properties() {
        return {
            success: { type: Boolean },
        }
    }

    constructor() {
        super()
    }

    connectedCallback() {
        super.connectedCallback()
        this.verifyCheckoutSession()
    }

    async onApiSuccess() {
        this.success = true

        await validateCurrentToken()

        setTimeout(() => {
            push(userHomePage(), true)
        }, 1500)
    }

    async onApiError(e) {
        const json = await e.detail.error.json()

        if (json.error) {
            showToast(json.error)
        }

        push('/')
    }

    verifyCheckoutSession() {
        const id = queryParam('session_id')
        this.api.post(`${this.api.baseRoute}/${id}`, {})
    }

    renderTitle() {
        if (this.success) {
            return t`Thank you for your payment!`
        }
        return t`Validating your payment...`
    }

    renderPage() {
        return html` ${this.renderLoader()}
        ${this.success
            ? html` <div class="success">
                  ${t`We have received your payment successfully`}
              </div>`
            : null}`
    }
}
customElements.define('qrcg-checkout-stripe-success', QrcgCheckoutStripeSuccess)
