import { html, css } from 'lit'

import '../ui/qrcg-box'

import '../payment/qrcg-paypal-button'

import { QRCGApiConsumer } from '../core/qrcg-api-consumer'

import { QRCGCheckoutController } from './qrcg-checkout-controller'

import '../payment/qrcg-stripe-button'

import '../ui/qrcg-tab'

import '../ui/qrcg-tab-content'

import { get } from '../core/api'

import { isEmpty, studlyCase } from '../core/helpers'

import { t } from '../core/translate'

import { QrcgCheckoutPage } from './qrcg-checkout-page'

import '../payment/qrcg-offline-payment-button'

import { price } from '../models/currency'

export class QrcgCheckout extends QrcgCheckoutPage {
    plans = new QRCGApiConsumer(this, 'subscription-plans')

    checkout = new QRCGCheckoutController(this)

    static styles = [
        super.styles,
        css`
            qrcg-paypal-button {
                max-width: 17rem;
                margin: auto;
                /* margin-top: 2rem; */
            }

            section {
                display: flex;
                align-items: center;
                justify-content: space-between;

                font-size: 1.3rem;
                line-height: 1.8;
                padding: 1rem 0;
                border-bottom: 2px dotted var(--gray-1);
            }

            .item {
                display: flex;
                color: var(--gray-2);
            }

            .item:nth-child(2) {
                font-weight: bold;
            }

            .hero {
                font-size: 6rem;
            }

            .tabs {
                display: flex;
            }

            .loading-or qrcg-loader {
                transform: translate(0, -23%) scale(0.5);
            }

            .loading-or {
                height: 2rem;
                display: inline-flex;
                align-items: center;
                justify-content: center;
            }
        `,
    ]

    static get properties() {
        return {
            planId: {},
            plan: { type: Object },
            paymentGateways: { type: Array },
        }
    }

    constructor() {
        super()
        this.plan = {}
    }

    async firstUpdated() {
        this.planId = this.checkout.loadPlanId()

        this.plan = await this.plans.get(this.planId)
    }

    connectedCallback() {
        super.connectedCallback()
        this.fetchEnabledPaymentGateways()
    }

    async fetchEnabledPaymentGateways() {
        const { response } = await get('payment-gateways')

        const json = await response.json()

        this.paymentGateways = json.data.filter((p) => p.enabled)
    }

    renderPaymentGatewaysTabs() {
        if (isEmpty(this.paymentGateways)) return

        return this.paymentGateways.map((p, i) => {
            return (() => {
                const _i = i

                return html`
                    <qrcg-tab
                        tab-id=${this.paymentGatewayTabId(p)}
                        .active=${_i === 0}
                    >
                        ${p.name}
                    </qrcg-tab>
                `
            })()
        })
    }

    renderPaymentGatewaysTabContents() {
        if (isEmpty(this.paymentGateways)) return this.renderLoader()

        return this.paymentGateways.map((p) => {
            try {
                const rendererName = `render${studlyCase(p.slug)}TabContent`

                return this[rendererName](p)
            } catch (ex) {
                return null
            }
        })
    }

    renderPaypalTabContent(paymentGateway) {
        return html`
            <qrcg-tab-content
                tab-id=${this.paymentGatewayTabId(paymentGateway)}
            >
                <template>
                    <qrcg-paypal-button
                        plan=${this.plan.id}
                    ></qrcg-paypal-button>
                </template>
            </qrcg-tab-content>
        `
    }

    renderStripeTabContent(paymentGateway) {
        return html`
            <qrcg-tab-content
                tab-id=${this.paymentGatewayTabId(paymentGateway)}
            >
                <template>
                    <qrcg-stripe-button></qrcg-stripe-button>
                </template>
            </qrcg-tab-content>
        `
    }

    renderOfflinePaymentGatewayTabContent(paymentGateway) {
        return html`
            <qrcg-tab-content
                tab-id=${this.paymentGatewayTabId(paymentGateway)}
            >
                <template>
                    <qrcg-offline-payment-button></qrcg-offline-payment-button>
                </template>
            </qrcg-tab-content>
        `
    }

    paymentGatewayTabId(paymentGateway) {
        return `payment-gateway-${paymentGateway.id}`
    }

    renderTitle() {
        return this.loadingOr(this.plan?.name, t`Checkout: ` + this.plan.name)
    }

    loadingOr(value, display) {
        display = display ? display : value

        if (value) {
            return display
        }

        return html` <div class="loading-or">
            <qrcg-loader> </qrcg-loader>
        </div>`
    }

    renderPage() {
        return html`
            <section>
                <div class="item">${t`Dynamic QR codes`}</div>
                <div class="item">
                    ${this.loadingOr(this.plan?.number_of_dynamic_qrcodes)}
                </div>
            </section>

            <section>
                <div class="item">${t`Scans`}</div>
                <div class="item">
                    ${this.loadingOr(this.plan?.number_of_scans)}
                </div>
            </section>

            <section>
                <div class="item">
                    ${this.loadingOr(
                        this.plan?.name,
                        this.plan.name + ' ' + t`Plan`
                    )}
                </div>
                <div class="item">${t`Yearly`}</div>
            </section>

            <section>
                <div class="item">${t`Subtotal`}</div>
                <div class="item">
                    ${price(
                        this.loadingOr(
                            this.plan?.monthly_price,
                            this.checkout.getYearlyPrice(this.plan)
                        )
                    )}
                </div>
            </section>

            <section>
                <div class="item">${t`Total`}</div>
                <div class="item">
                    ${price(
                        this.loadingOr(
                            this.plan?.monthly_price,
                            this.checkout.getYearlyPrice(this.plan)
                        )
                    )}
                </div>
            </section>

            ${this.renderPaymentGatewaysTabs()}
            ${this.renderPaymentGatewaysTabContents()}
        `
    }
}

customElements.define('qrcg-checkout', QrcgCheckout)
