import { LitElement, html, css } from 'lit'
import { get } from '../core/api'
import { isEmpty } from '../core/helpers'
import { t } from '../core/translate'

import '../ui/qrcg-loader-h'

export class QrcgRelationSelect extends LitElement {
    static styles = [
        css`
            :host {
                display: block;
            }

            qrcg-select {
                display: block;
            }

            .loading-container {
                position: relative;
            }

            .loading-container qrcg-select {
                opacity: 0.5;
                pointer-events: none;
            }

            qrcg-loader-h {
                position: absolute;
                color: black;
                --qrcg-loader-h-color: black;
                transform: translateX(-50%) scale(0.5);
                top: 0;
                left: 50%;
            }
        `,
    ]

    static get properties() {
        return {
            endpoint: {},
            data: {},
            name: {},
            value: {},
            errors: { type: Array },
        }
    }

    connectedCallback() {
        super.connectedCallback()
        this.fetch()
    }

    async fetch() {
        const { response } = await get(`${this.endpoint}?list_all=true`)

        const json = await response.json()

        if (json instanceof Array) {
            this.data = json
        } else if (json.data instanceof Array) {
            this.data = json.data
        }
    }

    renderLoader() {
        return html`
            <div class="loading-container">
                <qrcg-select>
                    <label slot="label"><slot></slot></label>
                </qrcg-select>
                <qrcg-loader-h></qrcg-loader-h>
            </div>
        `
    }

    renderItem(item) {
        if (typeof item === 'object') {
            return html`<option value=${item.id}>${item.name}</option>`
        }

        return html`<option value=${item}>${t(item)}</option>`
    }

    render() {
        if (isEmpty(this.data)) {
            return this.renderLoader()
        }

        return html`
            <qrcg-select
                name=${this.name}
                value=${this.value}
                .errors=${this.errors}
            >
                <label slot="label"><slot></slot></label>
                ${this.data.map((item) => this.renderItem(item))}
            </qrcg-select>
        `
    }
}
customElements.define('qrcg-relation-select', QrcgRelationSelect)
