import { LitElement, html, css } from 'lit'

export class QrcgDashboardNotice extends LitElement {
    static styles = [
        css`
            :host {
                display: none;
                padding: 0.5rem;
                text-align: center;
                font-size: 0.8rem;
                background-color: var(--danger);
                color: white;
                position: relative;
                margin-top: -1rem;
                margin-left: -1rem;
                margin-right: -1rem;
                margin-bottom: 1rem;
                user-select: none;
                -webkit-user-select: none;
                line-height: 1.8;
                animation: notice-in 0.3s ease-in both;
            }

            :host([visible]) {
                display: block;
            }

            a {
                opacity: 0;
                position: absolute;
                top: 0;
                right: 0;
                left: 0;
                bottom: 0;
            }

            @keyframes notice-in {
                0% {
                    transform: translate(0, -2rem);
                    opacity: 0;
                }

                80% {
                    opacity: 0.4;
                }

                100% {
                    transform: translate(0, 0);
                    opacity: 1;
                }
            }
        `,
    ]

    constructor() {
        super()
        this.apiEvents = []
        this.visible = false
        this.onShowRequested = this.onShowRequested.bind(this)
        this.onHideRequested = this.onHideRequested.bind(this)
    }

    static get properties() {
        return {
            link: {},
            message: {},
            visible: { type: Boolean, reflect: true },
        }
    }

    connectedCallback() {
        super.connectedCallback()

        this.addApiEvent('request-show', this.onShowRequested)

        this.addApiEvent('request-hide', this.onHideRequested)

        this.registerApiEvents()
    }

    disconnectedCallback() {
        super.disconnectedCallback()
        this.removeApiEvents()
    }

    addApiEvent(name, cp) {
        this.apiEvents.push({ name, cp })
    }

    registerApiEvents() {
        for (const { name, cp } of this.apiEvents) {
            document.addEventListener(this.eventName(name), cp)
        }
    }

    removeApiEvents() {
        for (const { name, cp } of this.apiEvents) {
            document.removeEventListener(this.eventName(name), cp)
        }
    }

    eventName(name) {
        return this.tagName.toLowerCase() + ':' + name
    }

    onShowRequested(e) {
        this.visible = true

        this.message = e.detail.message

        this.link = e.detail.link
    }

    onHideRequested() {
        this.visible = false
        this.reset()
    }

    reset() {
        this.message = ''

        this.link = null
    }

    render() {
        let link = ''

        if (this.link) {
            link = html`<a href="${this.link}"></a>`
        }

        return html`${link}${this.message}`
    }
}
customElements.define('qrcg-dashboard-notice', QrcgDashboardNotice)

export const showNotice = ({ message, link }) => {
    document.dispatchEvent(
        new CustomEvent('qrcg-dashboard-notice:request-show', {
            detail: {
                message,
                link,
            },
        })
    )
}
