import { LitElement, html, css } from 'lit'

import '../ui/qrcg-button'

import '../ui/qrcg-icon'

import '../ui/qrcg-input'

import '../ui/qrcg-form'

import { mdiDownload } from '@mdi/js'

import { state } from './state'

import { observeState } from 'lit-element-state'

import { QRCodeModel } from '../models/qrcode-model'

import { QRCGFormController } from '../core/qrcg-form-controller'

import { debounce } from '../core/helpers'

import { SvgPngConverter } from '../common/qrcg-svg-png-converter'

import { t } from '../core/translate'

class QRCGDownloadQRCode extends observeState(LitElement) {
    formController = new QRCGFormController(this)
    static get styles() {
        return css`
            :host {
                display: block;
            }
            .buttons {
                display: grid;
                grid-gap: 1rem;
            }

            qrcg-button {
                width: 100%;
            }

            qrcg-button::part(button) {
                font-size: 2rem;
            }

            .button-content {
                display: flex;
                align-items: center;
                justify-content: center;
            }

            qrcg-icon {
                margin-right: 1rem;
                width: 3rem;
                height: 3rem;
            }

            h2 {
                text-align: center;
                color: var(--gray-2);
                font-size: 2rem;
                font-weight: 300;
            }

            qrcg-form {
                max-width: 400px;
                margin: auto;
                text-align: center;
            }

            .name-label {
                color: var(--gray-2);
                font-weight: bold;
            }

            @media (min-width: 900px) {
                qrcg-button::part(button) {
                    padding: 2rem 0;
                    font-size: 3rem;
                }

                .buttons {
                    max-width: 60%;
                    margin: auto;
                    grid-template-columns: repeat(2, 1fr);
                }

                h2 {
                    font-size: 3rem;
                }
            }
        `
    }

    static get properties() {
        return {
            data: {},
        }
    }

    constructor() {
        super()
        this.data = {}
        this.fireSave = this.fireSave.bind(this)
    }

    connectedCallback() {
        super.connectedCallback()

        this.data = { ...state.remoteRecord }

        this.fireSave = debounce(this.fireSave, 1000)
    }

    fireSave() {
        state.name = this.data.name

        this.dispatchEvent(
            new CustomEvent('qrcg-download-qrcode:name-change', {
                composed: true,
                bubbles: true,
            })
        )
    }

    href(type) {
        if (!state.remoteRecord) return ''

        const model = new QRCodeModel(state.remoteRecord)

        return model.getFileUrl(type)
    }

    makeDownloadName(type) {
        return (
            state.name
                .replace(/https?/s, '')
                .replace(/:/g, '')
                .replace(/\//g, '')
                .replace(/\./g, '_') +
            '.' +
            type
        )
    }

    pngClick() {
        const model = new QRCodeModel(state.remoteRecord)

        const url = model.getFileUrl('svg')

        const converter = new SvgPngConverter(url, model.getName() + '.png')

        converter.downloadPng()
    }

    render() {
        return html`
            <qrcg-form>
                <label class="name-label">${t`Give it a name`}</label>
                <qrcg-input
                    name="name"
                    placeholder=${t`QR Code name`}
                    @on-input=${this.fireSave}
                ></qrcg-input>
            </qrcg-form>

            <h2>${t`Your download is here`}</h2>

            <div class="buttons">
                <qrcg-button
                    href=${this.href('svg')}
                    target="_blank"
                    .loading=${false}
                    download=${this.makeDownloadName('svg')}
                >
                    <div class="button-content">
                        <qrcg-icon mdi-icon=${mdiDownload}></qrcg-icon>
                        SVG
                    </div>
                </qrcg-button>

                <qrcg-button @click=${this.pngClick} .loading=${false}>
                    <div class="button-content">
                        <qrcg-icon mdi-icon=${mdiDownload}></qrcg-icon>
                        PNG
                    </div>
                </qrcg-button>
            </div>
        `
    }
}

window.customElements.define('qrcg-download-qrcode', QRCGDownloadQRCode)
