import { LitElement, html, css } from 'lit'
import { classMap } from 'lit/directives/class-map.js'

class QRCGQRCodeFormSteps extends LitElement {
    static get styles() {
        return css`
            :host {
                display: block;
                user-select: none;
                -webkit-user-select: none;
                box-sizing: border-box;

                --inactive-bg: var(--gray-0);

                --active-bg: var(--primary-0);

                --passed-color: black;
            }

            * {
                box-sizing: border-box;
            }

            .progress-bar {
                list-style: none;
                overflow: hidden;
                font-weight: 600;
                display: none;
                counter-reset: li;
                padding: 0;
            }

            h2 {
                font-weight: 300;
                color: var(--gray-2);
            }

            @media (min-width: 1150px) {
                .progress-bar {
                    display: flex;
                }

                h2 {
                    display: none;
                }
            }

            .progress-bar__steps {
                background-color: var(--inactive-bg);
                color: var(--gray-2);
                width: 33%;
                position: relative;
                cursor: default;
                list-style-image: none;
                list-style-type: none;
                padding: 20px 5px;
                text-align: center;
                display: flex;
            }

            .progress-bar__steps--text {
                overflow: hidden;
                text-overflow: ellipsis;
                white-space: nowrap;
            }

            .progress-bar__steps {
                padding: 20px 0 20px 65px;
                text-align: left;
            }

            .progress-bar__steps:first-child {
                padding: 20px 0 20px 30px;
            }

            .progress-bar__steps:after {
                border-bottom: 50px solid transparent;
                border-top: 50px solid transparent;
                content: ' ';
                display: block;
                height: 0;
                left: 100%;
                margin-top: -50px;
                position: absolute;
                top: 50%;
                width: 0;
                border-left: 30px solid var(--inactive-bg);
                z-index: 2;
            }

            .progress-bar__steps:before {
                border-bottom: 50px solid transparent;
                border-top: 50px solid transparent;
                content: ' ';
                display: block;
                height: 0;
                left: 100%;
                margin-top: -50px;
                position: absolute;
                top: 50%;
                width: 0;
                border-left: 30px solid #fff;
                z-index: 1;
                margin-left: 5px;
            }

            .progress-bar .current {
                background: var(--active-bg);
                color: #fff;
            }

            .progress-bar .current:after {
                border-left: 30px solid var(--active-bg);
            }

            .progress-bar .passed {
                color: var(--passed-color);
            }

            .progress-bar .passed,
            .progress-bar .current {
                cursor: pointer;
            }

            .progress-bar__steps--numbers:before {
                content: counter(li) ' ';
                counter-increment: li;
                margin-right: 15px;
                background: transparent;
                border: 1px solid currentColor;
                border-radius: 50%;
                display: inline-block;
                height: 20px;
                line-height: 20px;
                text-align: center;
                width: 20px;
            }

            .current .progress-bar__steps--numbers:before {
                background: white;
                color: var(--active-bg);
            }
        `
    }

    static get properties() {
        return {
            steps: {
                type: Array,
            },
            currentStep: {},
        }
    }

    constructor() {
        super()

        this.steps = []
    }

    _onStepClick(e) {
        const step = e.currentTarget.getAttribute('step')

        if (!this._isStepPassed(step)) {
            return
        }

        this.dispatchEvent(
            new CustomEvent('request-step-change', {
                detail: { step },
            })
        )
    }

    _isStepPassed(step) {
        if (typeof step === 'string') {
            step = this.steps.find((s) => s.value === step)
        }

        const currentStep = this.steps.find(
            (step) => step.value === this.currentStep
        )

        return this.steps.indexOf(step) < this.steps.indexOf(currentStep)
    }

    get currentStepName() {
        return this.steps.find((s) => s.value === this.currentStep).name
    }

    render() {
        return html`
            <h2>${this.currentStepName}</h2>
            <ol class="progress-bar">
                ${this.steps.map(
                    (step) => html`
                        <li
                            class="progress-bar__steps ${classMap({
                                'not-current': step.value !== this.currentStep,
                                current: step.value === this.currentStep,
                                passed: this._isStepPassed(step),
                            })}"
                            step=${step.value}
                            @click=${this._onStepClick}
                        >
                            <span class="progress-bar__steps--numbers"></span>
                            <span class="progress-bar__steps--text"
                                >${step.name}</span
                            >
                        </li>
                    `
                )}
            </ol>
        `
    }
}

window.customElements.define('qrcg-qrcode-form-steps', QRCGQRCodeFormSteps)
