import { LitElement, css } from 'lit'

import { html, unsafeStatic } from 'lit/static-html.js'

import '../core/qrcg-protected-route'

import '../dashboard/qrcg-dashboard-layout'

import './qrcg-qrcode-type-selector'

import { state, defaultState } from './state'

import { observeState } from 'lit-element-state'

import '../ui/qrcg-button'

import './qrcg-qrcode-form-steps'

import './qrcg-qrcode-designer'

import './qrcg-download-qrcode'

import { capitalize, equals, isEmpty, only, titleCase } from '../core/helpers'

import { push } from '../core/qrcg-router'

import { QRCGApiConsumer } from '../core/qrcg-api-consumer'

import { QRCGTitleController } from '../core/qrcg-title-controller'

import { QRCGRouteParamsController } from '../core/qrcg-route-params-controller'

import { QrcgDashboardBreadcrumbs } from '../dashboard/qrcg-dashboard-breadcrumbs'

import { t } from '../core/translate'
import { confirm } from '../ui/qrcg-confirmation-modal'

class QRCGQRCodeForm extends observeState(LitElement) {
    api = new QRCGApiConsumer(this, 'qrcodes', 'qrcg-button')

    titleController = new QRCGTitleController(this)

    routeParams = new QRCGRouteParamsController(this)

    static get styles() {
        return css`
            :host {
                display: block;
            }

            .step-navigator {
                display: flex;
                justify-content: space-between;
                margin-bottom: 1rem;
            }

            .step-navigator.bottom {
                margin-top: 1rem;
                margin-bottom: 0;
            }

            *[hidden] {
                display: none;
            }
        `
    }

    static get properties() {
        return {
            steps: { type: Array },
            loaders: { state: true },
        }
    }

    constructor() {
        super()

        this.updatePageTitle = this.updatePageTitle.bind(this)

        this.save = this.save.bind(this)

        this.steps = [
            {
                name: t('Select a type'),
                value: 'type',
            },
            {
                name: t('Data entry'),
                value: 'data',
            },
            {
                name: t('Design QR Code'),
                value: 'design',
            },
            { name: t('Download'), value: 'download' },
        ]
    }

    connectedCallback() {
        super.connectedCallback()

        this.addEventListener('before-step-change', this._onBeforeStepChange)

        this.addEventListener('after-step-change', this._onAfterStepChange)

        this.addEventListener('api:success', this._onApiSuccess)

        this.addEventListener('api:before-request', this._onBeforeRequest)

        this.addEventListener('api:after-request', this._onAfterRequest)

        this.addEventListener('qrcg-download-qrcode:name-change', this.save)

        this.addEventListener('on-input', this.onInput)

        this.addEventListener(
            'qrcg-img-selector:disabled-option-click',
            this.onImgSelectorDisabledOptionClick
        )

        window.addEventListener(
            'qrcg-router:location-changed',
            this.onLocationChanged
        )

        const id = this.routeParams.get('id')

        this.fetch(id)

        this.updatePageTitle()

        this.resetState()

        this.updateBreadcrumbs()
    }

    disconnectedCallback() {
        this.removeEventListener('before-step-change', this._onBeforeStepChange)

        this.removeEventListener('after-step-change', this._onAfterStepChange)

        this.removeEventListener('api:success', this._onApiSuccess)

        this.removeEventListener('api:before-request', this._onBeforeRequest)

        this.removeEventListener('api:after-request', this._onAfterRequest)

        this.removeEventListener('qrcg-download-qrcode:name-change', this.save)

        this.removeEventListener('on-input', this.onInput)

        window.removeEventListener(
            'qrcg-router:location-changed',
            this.onLocationChanged
        )

        this.removeEventListener(
            'qrcg-img-selector:disabled-option-click',
            this.onImgSelectorDisabledOptionClick
        )
    }

    onImgSelectorDisabledOptionClick(e) {
        if (e.detail.name === 'shape' || e.detail.name === 'advancedShape') {
            confirm({
                title: 'Upgrade Required',
                message: t`You need to upgrade your subscription to use this feature.`,
                affirmativeText: t`OK`,
                negativeText: null,
            })
        }
    }

    onLocationChanged = () => {
        this.updatePageTitle()
        this.updateBreadcrumbs()
    }

    onInput = async (e) => {
        if (e.composedPath()[0].tagName === 'QRCG-FILE-INPUT') {
            // Actually we just need to get the related file model,
            // So cache local record, reload and then re apply cached local record.

            const localRecord = {
                ...this.localRecord,
            }

            await this.fetch(state.id)

            this.localRecord = localRecord
        }

        if (e.detail.name === 'qrcg-qrcode-designer') {
            state.design = e.detail.value
        }
    }

    updateBreadcrumbs() {
        const pathLinks =
            QrcgDashboardBreadcrumbs.buildBreadcrumbFromCurrentPath((part) => {
                if (part.match(/qrcodes/)) return 'QR Codes'

                return titleCase(part)
            })

        if (pathLinks.length === 0) return

        if (pathLinks[pathLinks.length - 1].href.match(/\d+/)) {
            pathLinks.splice(pathLinks.length - 1)
        }

        const lastLink = pathLinks[pathLinks.length - 1]

        pathLinks[pathLinks.length - 1] = {
            ...lastLink,
            text: this.titleController.pageTitle,
        }

        QrcgDashboardBreadcrumbs.setLinks(pathLinks)
    }

    async fetch(id) {
        if (!id) return

        const recordId = id ? id : state.id

        if (!isEmpty(recordId)) await this.api.get(recordId)
    }

    resetState() {
        Object.keys(defaultState).forEach((key) => {
            state[key] = defaultState[key]
        })
    }

    updatePageTitle() {
        if (window.location.pathname.match('edit')) {
            this.titleController.pageTitle = t('Edit QR Code')
        } else {
            this.titleController.pageTitle = t('Create QR Code')
        }
    }

    get loading() {
        return this.loaders > 0
    }

    set loading(loading) {
        this.loaders = loading ? this.loaders - 1 : this.loaders + 1
    }

    get localRecord() {
        return {
            id: state.id,
            data: state.data,
            type: state.type,
            design: state.design,
            name: state.name,
        }
    }

    set localRecord(record) {
        state.id = record.id
        state.data = record.data
        state.type = record.type
        state.name = record.name

        if (!isEmpty(record.design)) state.design = record.design
    }

    _onBeforeRequest = () => {
        state.loading = true
    }

    _onAfterRequest = () => {
        state.loading = true
    }

    _onTypeSelected(e) {
        state.type = e.detail.type

        this.changeCurrentStep('data')
    }

    _onStepChangeRequested = (e) => {
        this.changeCurrentStep(e.detail.step)
    }

    _onBeforeStepChange = (e) => {
        if (e.detail.currentStep === 'data') {
            state.data = this.form().data
        }

        if (e.detail.currentStep !== 'type') {
            this.beforeStepChangePromise = this.save()
        }
    }

    _onAfterStepChange = async (e) => {
        await new Promise((resolve) => setTimeout(resolve, 0))

        if (e.detail.currentStep === 'data') {
            this.form().data = state.data
        }
    }

    _onApiSuccess = async (e) => {
        const localId = this.localRecord.id

        this.localRecord = e.detail.response

        state.remoteRecord = e.detail.response

        const redirect = `/dashboard/qrcodes/edit/${this.localRecord.id}`

        if (isEmpty(localId) && window.location.pathname !== redirect) {
            push(redirect)
        }
    }

    save() {
        if (!this.shouldSave()) {
            return
        }

        return this.api.save({
            id: state.id,
            data: state.data,
            type: state.type,
            design: state.design,
            name: state.name,
        })
    }

    shouldSave() {
        const remoteRecord = only(
            this.api.record,
            Object.keys(this.localRecord)
        )

        if (equals(remoteRecord, this.localRecord)) return false

        if (isEmpty(this.localRecord.data)) {
            return false
        }

        return true
    }

    async changeCurrentStep(nextStep) {
        this.dispatchEvent(
            new CustomEvent('before-step-change', {
                detail: {
                    currentStep: state.currentStep,
                    nextStep,
                },
            })
        )

        if (this.beforeStepChangePromise) await this.beforeStepChangePromise

        const prevStep = state.currentStep

        state.currentStep = nextStep

        this.dispatchEvent(
            new CustomEvent('after-step-change', {
                detail: {
                    prevStep,
                    currentStep: state.currentStep,
                },
            })
        )
    }

    form() {
        return this.renderRoot.querySelector(`[role=form]`)
    }

    get downloadStep() {
        return this.renderRoot.querySelector('qrcg-download-qrcode')
    }

    navigateNext() {
        this.navigate(1)
    }

    navigateBack() {
        this.navigate(-1)
    }

    navigate(offset) {
        const currentIndex = this.steps.indexOf(
            this.steps.find((s) => s.value === state.currentStep)
        )

        const nextIndex = currentIndex + offset

        if (nextIndex === this.steps.length) {
            return this.onClose()
        }

        this.changeCurrentStep(this.steps[nextIndex].value)
    }

    onClose() {
        push('/dashboard/qrcodes')
    }

    renderStepType() {
        return html`
            <qrcg-qrcode-type-selector
                value=${state.type}
                @qrcode-type-selected=${this._onTypeSelected}
            ></qrcg-qrcode-type-selector>
        `
    }

    renderStepData() {
        let tag = `qrcg-${state.type}-form`

        tag = `<${tag} role="form"></${tag}>`

        return html`${unsafeStatic(tag)}`
    }

    renderStepDesign() {
        return html`<qrcg-qrcode-designer
            .remoteRecord=${state.remoteRecord}
            .design=${state.design}
            .type=${state.type}
            .data=${state.data}
            enable-large-preview
        ></qrcg-qrcode-designer>`
    }

    renderStepDownload() {
        return html`<qrcg-download-qrcode></qrcg-download-qrcode>`
    }

    renderCurrentStep() {
        const rendererName = `renderStep${capitalize(state.currentStep)}`

        return this[rendererName]()
    }

    renderNavigation(position) {
        return html`
            <div
                class="step-navigator ${position}"
                .hidden=${state.currentStep === 'type'}
            >
                <qrcg-button @click=${this.navigateBack}
                    >${t`Back`}</qrcg-button
                >
                <qrcg-button @click=${this.navigateNext}
                    >${this.nextText()}</qrcg-button
                >
            </div>
        `
    }

    nextText() {
        if (state.currentStep === 'download') return t('Close')

        return t('Next')
    }

    render() {
        return html`
            <qrcg-qrcode-form-steps
                @request-step-change=${this._onStepChangeRequested}
                currentStep=${state.currentStep}
                .steps=${this.steps}
            ></qrcg-qrcode-form-steps>

            ${this.renderNavigation('top')}
            <!-- New line -->
            ${this.renderCurrentStep()}
            <!-- New line -->
            ${this.renderNavigation('bottom')}
        `
    }
}

window.customElements.define('qrcg-qrcode-form', QRCGQRCodeForm)
