import { LitElement, html, css } from 'lit'

import { classMap } from 'lit/directives/class-map.js'
import { SvgPngConverter } from '../common/qrcg-svg-png-converter'

import { isEmpty } from '../core/helpers'

import { Config } from '../core/qrcg-config'

import '../ui/qrcg-loader'

import { iPhoneSafari } from '../core/helpers'

export class QRCG_QRcodeImage extends LitElement {
    get defaultImage() {
        return 'assets/images/default-qrcode-placeholder.png'
    }

    static get styles() {
        return css`
            :host {
                display: block;
                position: relative;
            }

            :host([preview-on-dbl-click]) {
                cursor: pointer;
            }

            .image {
                padding-bottom: 100%;
                width: 100%;
                background-position: center;
                background-size: contain;
                background-repeat: no-repeat;
                position: relative;
                transition: opacity 0.5s ease-in-out;
            }

            .instructions {
                color: var(--gray-2);
                font-size: 0.8rem;
                text-align: center;
                margin-top: 1rem;
            }

            qrcg-loader {
                position: absolute;
                top: 50%;
                left: 50%;
                transform: translateX(-50%) translateY(-50%) scale(0.8);
            }

            .inactive {
                opacity: 0.5;
            }

            .loading {
                opacity: 0.1;
            }
        `
    }

    static get properties() {
        return {
            url: {},
            imageUrl: {
                state: true,
            },
            loading: {
                type: Boolean,
                state: true,
            },
            previewOnDblClick: {
                type: Boolean,
                attribute: 'preview-on-dbl-click',
                reflect: true,
            },
            converToPng: {
                type: Boolean,
                attribute: 'convert-to-png',
            },
        }
    }

    constructor() {
        super()

        this.loaders = 0
        this.clickCount = 0
        this.converToPng = false
    }

    connectedCallback() {
        super.connectedCallback()

        this.addEventListener('click', this.onClick)
    }

    disconnectedCallback() {
        super.disconnectedCallback()
        this.removeEventListener('click', this.onClick)
        URL.revokeObjectURL(this.imageUrl)
    }

    onClick = () => {
        this.clickCount++

        this.dblClickHandle = setTimeout(() => {
            this.clickCount = 0
        }, 500)

        if (this.clickCount > 1) {
            if (!this.previewOnDblClick) {
                return
            }

            this.clickCount = 0

            const params = {
                src: this.url,
            }

            const searchParams = new URLSearchParams(params).toString()

            window.open(
                Config.get('app.url') +
                    '/dashboard/qrcodes/designer/preview?' +
                    searchParams,
                '_blank'
            )
        }
    }

    updated(changed) {
        if (changed.has('url')) {
            if (iPhoneSafari()) {
                // It appears mobile safari cannot convert
                // images efficiently on fly to png (espicially when batch loaded)
                // Batch converting to PNG causes the website to crash.
                // Safari has built in SVG rasterizing
                this._loadImage()
            } else {
                //
                if (!this.converToPng) this._loadImage()
                else this.loadConvertedPng()
            }
        }
    }

    async loadConvertedPng() {
        // Clear previously loaded image

        URL.revokeObjectURL(this.imageUrl)

        await new Promise((resolve) => setTimeout(resolve, 0))

        if (isEmpty(this.url)) {
            this.imageUrl = null
            return
        }

        this.loading = ++this.loaders > 0

        const converter = new SvgPngConverter(this.url)

        await converter.loadSvg()

        const data = await converter.getPngBlob(300, 300)

        const url = URL.createObjectURL(data)

        const img = new Image()

        img.onload = (() => {
            const _url = url
            return () => {
                this.imageUrl = _url

                this.loading = --this.loaders > 0

                setTimeout(() => img.remove(), 50)
            }
        })()

        img.src = url
    }

    _loadImage() {
        if (isEmpty(this.url)) {
            this.imageUrl = null
            return
        }

        this.loading = ++this.loaders > 0

        const img = new Image()

        img.onload = (() => {
            // keep last requested url as local variable
            const _url = this.url

            return () => {
                this.loading = --this.loaders > 0

                // only the last loaded image will be rendered

                setTimeout(() => {
                    this.imageUrl = _url
                }, 0)

                setTimeout(() => {
                    img.remove()
                }, 100)
            }
        })()

        img.src = this.url
    }

    render() {
        const url = !isEmpty(this.imageUrl) ? this.imageUrl : this.defaultImage

        return html`
            <div
                class=${classMap({
                    image: true,
                    inactive: isEmpty(this.imageUrl),
                    loading: this.loading,
                })}
                style="background-image:url(${url})"
            ></div>
            ${this.loading ? html`<qrcg-loader></qrcg-loader>` : html``}
            ${this.previewOnDblClick
                ? html`<div class="instructions">Double click to enlarge</div>`
                : null}
        `
    }
}

window.customElements.define('qrcg-qrcode-image', QRCG_QRcodeImage)
