import { LitElement, html, css } from 'lit'

import './qrcg-qrcode-row'

import { QRCGQRCodeListController } from './qrcg-qrcode-list-controller'

import { QRCGTitleController } from '../core/qrcg-title-controller'

import { isEmpty, queryParam, capitalize, debounce } from '../core/helpers'

import { QRCodeModel } from '../models/qrcode-model'

import './qrcg-qrcode-row'

import '../ui/qrcg-pagination'

import '../ui/qrcg-button'

import '../ui/qrcg-confirmation-modal'

import '../ui/qrcg-loader'

import { QrcgDashboardBreadcrumbs } from '../dashboard/qrcg-dashboard-breadcrumbs'

import { t } from '../core/translate'
import { push } from '../core/qrcg-router'
import { confirm } from '../ui/qrcg-confirmation-modal'

class QRCodeList extends LitElement {
    list = new QRCGQRCodeListController(this)

    titleController = new QRCGTitleController(this)

    static get styles() {
        return css`
            :host {
                display: block;
            }

            .toolbar {
                display: flex;
                align-items: center;
                margin-bottom: 1rem;
            }

            qrcg-pagination {
                margin-bottom: 1rem;
            }

            qrcg-qrcode-row {
                margin-bottom: 1rem;
            }

            .loading-container {
                display: flex;
                align-items: center;
                justify-content: center;
                margin-bottom: 1rem;
            }

            .empty-message {
                text-align: center;
                color: var(--gray-2);
            }

            qrcg-form.search::part(form) {
                display: flex;
                align-items: center;
                margin-bottom: 1rem;
            }

            .search-label {
                margin-right: 1rem;
                margin-left: auto;
                font-weight: bold;
            }
        `
    }

    static get properties() {
        return {
            records: { type: Array },
            paginated: { type: Object },
            page: {},
            archived: {},
            loading: { type: Boolean },
            disabled: { type: Boolean },
            name: {},
        }
    }

    constructor() {
        super()
        this.onLocationChanged = this.onLocationChanged.bind(this)
        this.onBeforeRequest = this.onBeforeRequest.bind(this)
        this.onAfterRequest = this.onAfterRequest.bind(this)
        this.onSearchChanged = this.onSearchChanged.bind(this)
    }

    connectedCallback() {
        super.connectedCallback()

        this.addEventListener('api:success', this.onApiSuccess)

        this.addEventListener('api:before-request', this.onBeforeRequest)

        this.addEventListener('api:after-request', this.onAfterRequest)

        this.addEventListener(
            'qrcg-qrcode-list:request-refresh',
            this.onRefreshRequested
        )

        this.addEventListener('qrcg-qrcode-row:before-copy', this.onBeforeCopy)

        this.addEventListener('qrcg-qrcode-row:after-copy', this.onAfterCopy)

        this.titleController.pageTitle = t`QR Code List`

        window.addEventListener(
            'qrcg-router:location-changed',
            this.onLocationChanged
        )

        this.onLocationChanged()

        this.fetchData()

        this.onSearchChanged = debounce(this.onSearchChanged, 1000)

        this.updateBreadcrumbs()
    }

    disconnectedCallback() {
        super.disconnectedCallback()

        this.removeEventListener('api:success', this.onApiSuccess)

        this.removeEventListener('api:before-request', this.onBeforeRequest)

        this.removeEventListener('api:after-request', this.onAfterRequest)

        window.removeEventListener(
            'qrcg-router:location-changed',
            this.onLocationChanged
        )

        this.removeEventListener(
            'qrcg-qrcode-list:request-refresh',
            this.onRefreshRequested
        )

        this.removeEventListener(
            'qrcg-qrcode-row:before-copy',
            this.onBeforeCopy
        )

        this.removeEventListener('qrcg-qrcode-row:after-copy', this.onAfterCopy)
    }

    updateBreadcrumbs() {
        const links = QrcgDashboardBreadcrumbs.buildBreadcrumbFromCurrentPath()

        links[links.length - 1] = {
            ...links[links.length - 1],
            text: t`QR Codes`,
        }

        QrcgDashboardBreadcrumbs.setLinks(links)
    }

    onLocationChanged() {
        const page = queryParam('page')

        this.archived = isEmpty(queryParam('archived')) ? false : true

        if (!isEmpty(page)) this.page = page
    }

    willUpdate(changed) {
        const dependencies = ['page', 'archived', 'name']

        const shouldFetch = dependencies.reduce((isChanged, att) => {
            return isChanged || changed.has(att)
        }, false)

        if (shouldFetch) {
            this.fetchData()
        }
    }

    onApiSuccess = (e) => {
        if (Object.keys(e.detail.response).indexOf('data') < 0) {
            return
        }

        if (!(e.detail.response.data instanceof Array)) {
            return
        }

        this.paginated = e.detail.response

        this.records = e.detail.response.data.map(
            (record) => new QRCodeModel(record)
        )
    }

    onBeforeRequest() {
        this.loading = true
    }

    onAfterRequest() {
        this.loading = false
    }

    onSearchChanged(e) {
        this.name = e.detail.value
    }

    onBeforeCopy = () => {
        this.disabled = true
    }

    onAfterCopy = () => {
        this.disabled = false

        if (this.page == 1) {
            this.fetchData()
        } else {
            const archived = this.archived ? '&archived=true' : ''

            push(`${window.location.pathname}?page=1${archived}`)
        }
    }

    fetchData() {
        const searchParams = {
            path: `${window.location.origin}${window.location.pathname}${
                this.archived ? '?archived=true' : ''
            }`,
            search_archived: this.archived,
        }

        if (!isEmpty(this.name)) {
            searchParams.search_name = this.name
        }

        if (this.page) {
            searchParams.page = this.page
        }

        this.list.search(searchParams)
    }

    async onToggleArchive(e) {
        const action = e.detail.qrcode.archived ? 'restore' : 'archive'

        try {
            await confirm({
                message: t`Are you sure you want to ${action} this QR Code?`,
                affirmativeText: t(capitalize(action)),
            })

            await this.list.archive(
                e.detail.qrcode.id,
                !e.detail.qrcode.archived
            )

            await this.fetchData()
        } catch {
            //
        }
    }

    renderRecords() {
        if (isEmpty(this.records)) return ''

        const mapped = this.records.map(
            (qrcode) =>
                html`
                    <qrcg-qrcode-row
                        .qrcode=${qrcode}
                        @on-toggle-archive=${this.onToggleArchive}
                        ?disabled=${this.disabled}
                    ></qrcg-qrcode-row>
                `
        )

        return mapped
    }

    shouldShowEmptyMessage() {
        return (
            this.records instanceof Array &&
            this.records.length === 0 &&
            !this.loading
        )
    }

    shouldShowPagination() {
        return !isEmpty(this.records) && !this.loading
    }

    render() {
        return html`
            <qrcg-form class="search">
                <label class="search-label">${t`Search`}</label>
                <qrcg-input
                    placeholder=${t`Search by name`}
                    type="search"
                    @on-input=${this.onSearchChanged}
                ></qrcg-input>
            </qrcg-form>

            ${this.loading
                ? html`
                      <div class="loading-container">
                          <qrcg-loader></qrcg-loader>
                      </div>
                  `
                : this.renderRecords()}
            ${this.shouldShowEmptyMessage()
                ? html`<h2 class="empty-message">
                      ${t`There are no records to show ...`}
                  </h2>`
                : ''}
            ${this.shouldShowPagination()
                ? html`<qrcg-pagination
                      .pagination=${this.paginated}
                  ></qrcg-pagination>`
                : ''}
        `
    }
}

window.customElements.define('qrcg-qrcode-list', QRCodeList)
