import { LitElement, html, css } from 'lit'

import { observeState } from 'lit-element-state'

import { classMap } from 'lit/directives/class-map.js'

import { t } from '../core/translate'

import { state } from './state'

import { isEmpty } from '../core/helpers'

import { QRCGRouteParamsController } from '../core/qrcg-route-params-controller'

import { qrTypes } from '../models/qr-types'
import { currentPlan } from '../core/subscription/logic'
import { isSuperAdmin } from '../core/auth'

class QRCGQRCodeTypeSelector extends observeState(LitElement) {
    routeParams = new QRCGRouteParamsController(this)

    static get styles() {
        return css`
            :host {
                display: block;
                box-sizing: border-box;
                padding-right: 0.3rem;

                --background-color: var(--gray-0);
                --border-color: var(--primary-0);
                --name-color: var(--gray-2);
            }

            * {
                box-sizing: border-box;
            }

            .types {
                display: grid;

                grid-gap: 1rem;
            }

            @media (min-width: 800px) {
                .types {
                    grid-template-columns: 1fr 1fr;
                }
            }

            .type {
                border-left: 4px solid var(--border-color);
                padding: 1rem;
                border-radius: 0.5rem;
                cursor: pointer;
                background-color: var(--background-color);
                user-select: none;
                -webkit-user-select: none;
                touch-action: manipulation;
                transition: 0.2s cubic-bezier(0.455, 0.03, 0.515, 0.955);
                position: relative;
            }

            .type[loading] {
                opacity: 0.5;
                pointer-events: none;
            }

            .type[disabled] {
                opacity: 0.5;
                pointer-events: none;
            }

            .type:focus {
                outline: 0.13rem solid var(--primary-0);
            }

            .type:hover,
            .type.selected:not([loading]) {
                box-shadow: 2px 2px 2px 0px rgba(0, 0, 0, 0.2);
            }

            .name {
                color: var(--name-color);
                margin-bottom: 0.5rem;
            }

            .type.selected:not([loading]) .name {
                color: black;
                font-weight: bold;
            }

            .cat {
                font-size: 0.8rem;
                color: var(--primary-0);
            }

            h2 {
                color: var(--gray-2);
                font-weight: 300;
            }

            .disabled-type-message {
                color: var(--danger);
                font-size: 0.8rem;
                position: absolute;
                top: 1rem;
                right: 1rem;
            }
        `
    }

    static get properties() {
        return {
            types: { type: Array },
            value: { type: String },
        }
    }

    constructor() {
        super()

        this.types = qrTypes

        this.currentPlan = currentPlan()
    }

    connectedCallback() {
        super.connectedCallback()
        window.scrollTo({ top: 0, behavior: 'auto' })
    }

    _typeClick(e) {
        const type = e.currentTarget.getAttribute('type').toLowerCase()

        this.value = type

        this.dispatchEvent(
            new CustomEvent(`qrcode-type-selected`, {
                detail: {
                    type,
                },
            })
        )
    }

    typeKeyPress(e) {
        if (e.key === 'Enter') {
            this._typeClick(e)
        }
    }

    isLoading() {
        return (
            isEmpty(state.remoteRecord) &&
            this.routeParams.get('id') != null &&
            !isNaN(this.routeParams.get('id'))
        )
    }

    currentPlanHasType(id) {
        if (isSuperAdmin()) return true

        return this.currentPlan?.qr_types?.find((t) => t == id)
    }

    render() {
        return html`
            <div class="types">
                ${this.types.map(
                    (type) => html`<div
                        tabindex="0"
                        class="type ${classMap({
                            selected: type.id === this.value,
                        })}"
                        @click=${this._typeClick}
                        @keypress=${this.typeKeyPress}
                        type=${type.id}
                        ?loading=${this.isLoading()}
                        ?disabled=${!this.currentPlanHasType(type.id)}
                    >
                        <div class="name">${t(type.name)}</div>
                        <div class="cat">${t(type.cat)}</div>
                        ${!this.currentPlanHasType(type.id)
                            ? html`
                                  <div class="disabled-type-message">
                                      ${t`Upgrade to use this type.`}
                                  </div>
                              `
                            : null}
                    </div>`
                )}
            </div>
        `
    }
}

window.customElements.define(
    'qrcg-qrcode-type-selector',
    QRCGQRCodeTypeSelector
)
