import { css, html } from 'lit'

import { QrcgDashboardForm } from '../dashboard/qrcg-dashboard-form'

import '../ui/qrcg-input'

import '../ui/qrcg-balloon-selector'

import { isEmpty } from '../core/helpers'

import { t } from '../core/translate'

import { qrTypes } from '../models/qr-types'

import { Config } from '../core/qrcg-config'

import { getFeatures } from '../models/features'

export class QrcgSubscriptionPlanForm extends QrcgDashboardForm {
    static get styles() {
        return [
            ...super.styles,
            css`
                :host {
                    padding: 0.1rem;
                }

                .frequency label {
                    font-weight: bold;
                    margin-right: 1rem;
                    user-select: none;
                    -webkit-user-select: none;
                    font-size: 0.8rem;
                }

                .frequency {
                    display: flex;
                    align-items: center;
                }

                .payment-gateways {
                    padding: 1rem;
                    background-color: var(--gray-0);
                    line-height: 1.7rem;
                }

                .payment-gateways-label {
                    font-weight: bold;
                    text-transform: capitalize;
                    margin-bottom: 1rem;
                    display: block;
                }

                .payment-gateway-entry {
                    display: flex;
                    flex-wrap: wrap;
                    border-bottom: 1px solid var(--gray-1);
                    padding-bottom: 0.5rem;
                    margin-bottom: 0.5rem;
                    font-size: 0.8rem;
                }

                .payment-gateway-entry:last-child {
                    margin-bottom: 0;
                    padding-bottom: 0;
                    border-bottom: 0;
                }

                .payment-gateway-key {
                    text-transform: uppercase;
                    margin-right: 1rem;
                    flex: 1;
                }

                .payment-gateway-value {
                    flex: 1;
                    user-select: text;
                    -webkit-user-select: text;
                }

                .payment-gateway-value span {
                    display: block;
                }
            `,
        ]
    }
    constructor() {
        super({
            apiBaseRoute: 'subscription-plans',
        })
    }

    connectedCallback() {
        super.connectedCallback()
        this.fetchOneTimePlanConfig()
    }

    static get properties() {
        return {
            ...super.properties,
            wplusIntegrationEnabled: { type: Boolean },
        }
    }

    async fetchOneTimePlanConfig() {
        this.wplusIntegrationEnabled = Config.get(
            'app.wplus_integration_enabled'
        )
    }

    renderPaymentGatewayFields() {
        let keys =
            'paypal_plan_id|paypal_product_id|stripe_price_id|stripe_product_id'.split(
                '|'
            )

        keys = keys.filter((key) => !isEmpty(this.data[key]))

        if (isEmpty(keys)) return

        const labels = keys.map((k) => k.replace(/_/g, ' '))

        const values = keys.map((k) =>
            isEmpty(this.data[k]) ? 'Not available' : this.data[k]
        )

        return html`
            <div class="payment-gateways">
                <div class="payment-gateways-label">
                    ${t`Payment gateway fields`}
                </div>

                ${keys.map(
                    (key, i) => html`
                        <div class="payment-gateway-entry">
                            <div class="payment-gateway-key">${labels[i]}</div>
                            <div class="payment-gateway-value">
                                <span>${values[i]}</span>
                            </div>
                        </div>
                    `
                )}
            </div>
        `
    }

    renderFormFields() {
        return html`
            <qrcg-input name="name" placeholder="${t`basic plan`}">
                ${t`Name`}
            </qrcg-input>

            ${this.wplusIntegrationEnabled
                ? html`
                      <qrcg-balloon-selector
                          name="frequency"
                          .options=${[
                              {
                                  name: t`One Time`,
                                  value: 'ONE_TIME',
                              },
                              {
                                  name: t`Yearly`,
                                  value: 'YEAR',
                              },
                          ]}
                      >
                          ${t`Frequency`}
                      </qrcg-balloon-selector>
                  `
                : null}
            ${!this.wplusIntegrationEnabled || this.data.frequency === 'YEAR'
                ? html`
                      <qrcg-input
                          name="monthly_price"
                          placeholder="3"
                          type="number"
                          min="0"
                      >
                          ${t`Monthly price (charged yearly)`}
                      </qrcg-input>
                  `
                : html`
                      <qrcg-input
                          name="price"
                          placeholder="Price"
                          type="number"
                          min="0"
                      >
                          ${t`Price`}
                      </qrcg-input>
                  `}

            <qrcg-input
                name="number_of_dynamic_qrcodes"
                type="number"
                placeholder="150"
            >
                ${t` Number of dynamic QR codes`}
            </qrcg-input>

            <qrcg-input name="number_of_scans" placeholder="10000">
                ${t`Number of scans`}
            </qrcg-input>

            <qrcg-balloon-selector
                name="qr_types"
                .options=${qrTypes.map((t) => ({ ...t, value: t.id }))}
                multiple
            >
                ${t`QR Code Types`}
            </qrcg-balloon-selector>

            <qrcg-balloon-selector
                name="features"
                .options=${getFeatures()}
                multiple
            >
                ${t`Features`}
            </qrcg-balloon-selector>

            <qrcg-checkbox name="is_popular"> ${t`Popular`} </qrcg-checkbox>
            <qrcg-checkbox name="is_hidden"> ${t`Hidden`} </qrcg-checkbox>
            <qrcg-checkbox name="is_trial"> ${t`Is Trial`} </qrcg-checkbox>

            ${this.data.is_trial
                ? html`
                      <qrcg-input
                          name="trial_days"
                          placeholder="15"
                          type="number"
                          min="0"
                          step="1"
                      >
                          ${t`Trial days`}
                      </qrcg-input>
                  `
                : null}
            ${this.renderPaymentGatewayFields()}
        `
    }
}

customElements.define('qrcg-subscription-plan-form', QrcgSubscriptionPlanForm)
