import { html, css } from 'lit'

import { unsafeHTML } from 'lit/directives/unsafe-html.js'

import { mdiAlertOctagon, mdiCheckboxMarked } from '@mdi/js'

import { isEmpty } from '../core/helpers'

import { QRCGApiConsumer } from '../core/qrcg-api-consumer'

import { t } from '../core/translate'

import { confirm } from '../ui/qrcg-confirmation-modal'

import { showToast } from '../ui/qrcg-toast'

import { QrcgSystemPage } from './qrcg-system-page'

export class SystemStatus extends QrcgSystemPage {
    api = new QRCGApiConsumer(this, 'system')

    static styles = [
        css`
            :host {
                display: block;
            }

            * {
                box-sizing: border-box;
            }

            .entry {
                display: flex;
                margin: 1rem 0 0 0;
                background-color: var(--gray-0);
                padding: 2rem;
                flex-direction: column;
            }

            qrcg-icon {
                display: none;
                width: 2rem;
                height: 2rem;
                margin: 0 1rem;
            }

            .success qrcg-icon.success {
                display: block;
                color: var(--success-0);
            }

            .fail qrcg-icon.fail {
                display: block;
                color: var(--danger);
            }

            .action-required qrcg-icon.action-required {
                display: block;
                color: var(--warning-1);
            }

            .title {
                font-weight: bold;
                min-width: 30%;
                margin: 0 0 1rem;
            }

            .text {
                font-weight: bold;
            }

            .text-and-information {
                display: flex;
                flex-direction: column;
                flex: 1;
                position: relative;
                line-height: 1.5;
            }

            .information {
                margin-top: 0.5rem;
                color: var(--gray-2);
            }

            .instructions {
                margin-top: 0.5rem;
            }

            .details {
                display: flex;
                flex: 1;
            }

            .action {
                padding-top: 1rem;
            }

            code {
                display: block;
                padding: 0.5rem;
                background-color: white;
                margin: 1rem 0 0 0;
                color: black;
                width: 100%;
                overflow: scroll;
                scrollbar-width: none;
                max-width: 55vw;
            }

            code::-webkit-scrollbar {
                display: none;
            }

            .loading-container {
                display: flex;
                align-items: center;
                justify-content: center;
                padding: 5rem;
            }

            @media (min-width: 50rem) {
                .entry {
                    flex-direction: row;
                }
                .action {
                    padding-top: 0;
                }
            }
        `,
    ]

    static get properties() {
        return {
            systemStatus: {
                type: Array,
            },
            loading: { type: Boolean },
            databaseUpdateAvailable: { type: Boolean },
        }
    }

    constructor() {
        super()

        this.loaders = 0
    }

    connectedCallback() {
        super.connectedCallback()

        this.fetchStatus()

        document.addEventListener(
            'qrcg-system-status:request-refresh',
            this.refresh
        )
    }

    disconnectedCallback() {
        super.disconnectedCallback()

        document.removeEventListener(
            'qrcg-system-status:request-refresh',
            this.refresh
        )
    }

    refresh = () => {
        this.fetchStatus()
    }

    pageTitle() {
        return t('System Status')
    }

    onBeforeRequest = () => {
        this.loading = ++this.loaders > 0
    }

    onAfterRequest = () => {
        setTimeout(() => {
            this.loading = --this.loaders > 0
        }, 0)
    }

    async fetchStatus() {
        const { entries } = await this.api.get('status')

        this.systemStatus = entries

        const { update_available } = await this.api.get('check_database_update')

        this.databaseUpdateAvailable = update_available
    }

    renderEntry(entry) {
        return html`
            <div class="entry ${entry.type}">
                <div class="title">${entry.title}</div>
                <div class="details">
                    <qrcg-icon
                        mdi-icon=${mdiCheckboxMarked}
                        class="success"
                    ></qrcg-icon>

                    <qrcg-icon
                        mdi-icon=${mdiAlertOctagon}
                        class="fail"
                    ></qrcg-icon>

                    <div class="text-and-information">
                        <div class="text">${entry.text}</div>
                        ${(!isEmpty(entry.information) &&
                            html`<div class="information">
                                ${unsafeHTML(entry.information)}
                            </div>`) ||
                        ''}
                        ${(!isEmpty(entry.instructions) &&
                            html`<div class="instructions">
                                ${unsafeHTML(entry.instructions)}
                            </div>`) ||
                        ''}
                    </div>
                </div>
            </div>
        `
    }

    renderDatabaseUpdateEntry() {
        const _class = this.databaseUpdateAvailable
            ? 'action-required'
            : 'success'

        return html`
            <div class="entry ${_class}">
                <div class="title">${t`Database Update`}</div>
                <div class="details">
                    <qrcg-icon
                        mdi-icon=${mdiCheckboxMarked}
                        class="success"
                    ></qrcg-icon>

                    <qrcg-icon
                        mdi-icon=${mdiAlertOctagon}
                        class="fail"
                    ></qrcg-icon>

                    <qrcg-icon
                        mdi-icon=${mdiAlertOctagon}
                        class="action-required"
                    ></qrcg-icon>

                    <div class="text-and-information">
                        <div class="text">
                            ${this.databaseUpdateAvailable
                                ? t`Update is available`
                                : t`No action required`}
                        </div>
                        <div class="information">
                            ${this.databaseUpdateAvailable
                                ? t`Backup your database before proceeding`
                                : t`Database is up to date, you can still seed the database to insert default records.`}
                        </div>
                    </div>
                </div>
                <div class="action">
                    <qrcg-button @click=${this.onUpdateDatabaseClick}>
                        ${this.databaseUpdateAvailable
                            ? t`Update Database`
                            : t`Seed Database`}
                    </qrcg-button>
                </div>
            </div>
        `
    }

    async onUpdateDatabaseClick() {
        try {
            await confirm({
                title: t`Confirmation`,
                message: t`Make sure to backup your database before proceeding. Any data loss after the upgrade cannot be undone.`,
            })
        } catch (ex) {
            return
        }

        try {
            await this.api.post('system/update_database')
            showToast(t`Database update completed.`)

            this.refresh()
        } catch (error) {
            showToast(
                t`Could not update database, check log file for more details.`
            )
        }
    }

    render() {
        if (this.loading) {
            return html`
                <div class="loading-container">
                    <qrcg-loader></qrcg-loader>
                </div>
            `
        }

        return html`
            ${this.renderDatabaseUpdateEntry()}
            <!-- -->
            ${this.systemStatus instanceof Array
                ? this.systemStatus.map(this.renderEntry)
                : ''}
        `
    }
}
customElements.define('qrcg-system-status', SystemStatus)
