import { html } from 'lit'
import { isEmpty, ucfirst } from '../core/helpers'
import { t } from '../core/translate'
import { QRCGDashboardList } from '../dashboard/qrcg-dashboard-list'
import { confirm } from '../ui/qrcg-confirmation-modal'
import { FileModel } from '../ui/qrcg-file-input/model'
import { showToast } from '../ui/qrcg-toast'

export class QrcgTransactionList extends QRCGDashboardList {
    constructor() {
        super({
            baseRoute: 'transactions',
            singularRecordName: 'Transaction',
            frontendFormUrl: '/dashboard/transactions/view/',
        })
    }

    static listColumns = [
        { key: 'id', label: 'ID', width: '2rem' },
        { key: 'formatted_amount', label: 'Amount' },
        { key: 'subscription.user.name', label: 'User' },
        { key: 'subscription.subscription_plan.name', label: 'Plan' },
        { key: 'payment_gateway.name', label: 'Source' },
        { key: 'status', label: 'Status' },
        { key: 'created_at', label: 'Date' },
        {
            key: 'actions',
            label: 'Actions',
            width: '7rem',
        },
    ]

    cellContentRenderer(row, column) {
        switch (column.key) {
            case 'formatted_amount': {
                if (!isEmpty(row.formatted_amount)) {
                    return row.formatted_amount
                }
                return `${row.currency} ${row.amount}`
            }

            default:
                return super.cellContentRenderer(row, column)
        }
    }

    approveTransaction = (e) => {
        this.approveOrReject(e, 'approve')
    }

    rejectTransaction = (e) => {
        this.approveOrReject(e, 'reject')
    }

    async approveOrReject(e, verb) {
        e.preventDefault()
        e.stopImmediatePropagation()

        const target = e.composedPath()[0]

        const transactionId = target.transactionId

        await confirm({
            message: t`Are you sure you want to ${verb} the transaction? The customer will be notified.`,
            affirmativeText: t`${ucfirst(verb)}`,
        })

        await this.api.post(`transactions/${transactionId}/${verb}`)

        this.fetchData()

        showToast(t`Transaction has been ${verb}ed successfully!`)
    }

    rowActions(row) {
        if (row.payment_gateway.slug != 'offline-payment-gateway') return '---'

        const file = new FileModel({ remote: row.payment_proof })

        return html`
            <div>
                <a href=${file.directLink()} target="_blank"
                    >${t`Payment Proof`}</a
                >
                <a
                    href="#"
                    .transactionId=${row.id}
                    @click=${this.approveTransaction}
                    >${t`Approve`}</a
                >
                <a
                    href="#"
                    .transactionId=${row.id}
                    @click=${this.rejectTransaction}
                    >${t`Reject`}</a
                >
            </div>
        `
    }

    searchPlaceholder() {
        return t('By anything')
    }
}

customElements.define('qrcg-transaction-list', QrcgTransactionList)
