import { LitElement, html, css } from 'lit'

import { debounce, isEmpty } from '../core/helpers'

export class BaseInput extends LitElement {
    static get styles() {
        return css`
            :host {
                display: flex;
                flex-direction: column;
                position: relative;
            }

            .title {
                font-size: 0.8rem;
                margin-bottom: 0.5rem;
                font-weight: bold;
                letter-spacing: 1px;
                user-select: none;
                -webkit-user-select: none;
            }

            input {
                font-size: 1rem;
                padding: 0.5rem 0.8rem;
                border: 0.1rem solid var(--gray-1);
                border-radius: 0.5rem;
                -webkit-tap-highlight-color: transparent;
                margin-bottom: 0.5rem;
                background-color: white;
            }

            input:focus {
                border-color: var(--gray-2);
                outline: 0;
            }

            input::placeholder {
                color: var(--gray-1);
                font-weight: bold;
                font-style: var(--input-placeholder-font-style, italic);
            }

            .instructions {
                font-size: 0.8rem;
                padding: 0.5rem;
                background-color: var(--gray-0);
                margin-bottom: 0.5rem;
            }

            :host(:not([has-instructions])) .instructions {
                display: none;
            }

            .error {
                color: var(--danger);
                position: absolute;
                font-size: 0.8rem;
                font-weight: bold;
                bottom: 0.5rem;
                transform: translateY(100%);
                animation: fade-in ease 1s both;
            }

            @keyframes fade-in {
                from {
                    opacity: 0;
                }

                to {
                    opacity: 1;
                }
            }
        `
    }

    static get properties() {
        return {
            type: {},
            value: {},
            placeholder: {},
            name: {},
            autofocus: { type: Boolean },
            errors: { type: Array },
            disabled: { type: Boolean, reflect: true },
            hasInstructions: {
                type: Boolean,
                reflect: true,
                attribute: 'has-instructions',
            },
            debounce: { type: Boolean },
            debounceDelay: { attribute: 'debounce-delay' },
        }
    }

    constructor() {
        super()
        this.errors = []
        this.fireOnInput = this._fireOnInput
        this.debounceDelay = 1000
    }

    connectedCallback() {
        super.connectedCallback()
    }

    firstUpdated() {
        this._autoFocus()

        this.setHasInstructions()
    }

    updated(changed) {
        if (changed.has('debounce')) {
            if (this.debounce) {
                this.fireOnInput = debounce(
                    this._fireOnInput.bind(this),
                    this.debounceDelay
                )
            } else {
                this.fireOnInput = this._fireOnInput
            }
        }
    }

    setHasInstructions() {
        const slot = this.renderRoot.querySelector('slot[name=instructions]')

        const nodes = Array.from(
            slot.assignedNodes({
                flatten: true,
            })
        )

        const text = nodes.reduce(
            (result, node) => result + node.textContent,
            ''
        )

        if (text.length > 0) {
            this.hasInstructions = true
        } else {
            this.hasInstructions = false
        }
    }

    _autoFocus() {
        if (this.autofocus) {
            this.shadowRoot.querySelector('input').focus()
        }
    }

    _input(e) {
        this.value = e.target.value

        this.fireOnInput()
    }

    _fireOnInput() {
        this.dispatchEvent(
            new CustomEvent('on-input', {
                bubbles: true,
                composed: true,

                detail: {
                    value: this.value,
                    name: this.name,
                },
            })
        )
    }

    renderInput() {
        return html`
            <input
                id="input-${this.name}"
                type=${this.type}
                @input=${this._input}
                placeholder=${this.placeholder}
                .value=${this.value || ''}
                part="input"
                .disabled=${this.disabled}
            />
        `
    }

    renderLabel() {
        return html`
            <label class="title" for="input-${this.name}" part="label">
                <slot></slot>
            </label>
        `
    }

    renderInstructions() {
        return html`
            <div class="instructions">
                <slot name="instructions"></slot>
            </div>
        `
    }

    renderErrors() {
        return !isEmpty(this.errors)
            ? html`<label class="error">${this.errors[0]}</label>`
            : html``
    }

    render() {
        return html`
            ${this.renderLabel()}
            <!-- New line -->
            ${this.renderInstructions()}
            <!-- New line -->
            ${this.renderInput()}
            <!-- New line -->
            ${this.renderErrors()}
        `
    }
}
