import { LitElement, html, css } from 'lit'
import { isEmpty } from '../core/helpers'

export class QrcgCodeInput extends LitElement {
    static styles = [
        css`
            :host {
                display: block;
                position: relative;
            }

            #editor-container {
                min-height: 450px;
                margin-bottom: 0.5rem;
            }

            label *::slotted(*) {
                display: block;
                margin-bottom: 1rem;
                font-size: 0.8rem;
                font-weight: bold;
                letter-spacing: 1px;
                user-select: none;
                -webkit-user-select: none;
            }

            .error {
                color: var(--danger);
                position: absolute;
                font-size: 0.8rem;
                font-weight: bold;
                bottom: 0.5rem;
                transform: translateY(100%);
                animation: fade-in ease 1s both;
            }

            .monaco-editor {
                padding-top: 1rem;
            }

            @keyframes fade-in {
                from {
                    opacity: 0;
                }

                to {
                    opacity: 1;
                }
            }

            qrcg-loader {
                position: absolute;
                top: 50%;
                left: 50%;
                z-index: 100;
                transform: translate(-50%, -50%);
            }
        `,
    ]

    static get properties() {
        return {
            name: {},
            value: {},
            errors: { type: Array },
            loading: { type: Boolean },
            language: {},
        }
    }

    constructor() {
        super()

        this.loading = true

        this.language = 'javascript'
    }

    connectedCallback() {
        super.connectedCallback()
    }

    disconnectedCallback() {
        super.disconnectedCallback()

        // Clearing up monaco
        window.monaco.editor.getModels().forEach((model) => model.dispose())
    }

    async firstUpdated() {
        await this.loadMonaco()

        window.require.config({
            paths: { vs: '/assets/lib/monaco-editor/min/vs' },
        })

        window.require(['vs/editor/editor.main'], () => {
            this.editor = window.monaco.editor.create(
                this.shadowRoot.querySelector('#editor-container'),
                {
                    theme: 'vs-dark',
                    value: '',
                    language: this.language,
                    fontSize: '18px',
                }
            )

            this.editor.getModel().onDidChangeContent(() => {
                this.fireOnInput()
            })

            this.loading = false

            this.syncEditorValue()
        })
    }

    updated(changed) {
        if (changed.has('value')) {
            this.syncEditorValue()
        }
    }

    syncEditorValue() {
        if (!this.editor) return

        if (this.value !== this.editor.getModel().getValue())
            this.editor.getModel().setValue(this.value)
    }

    fireOnInput() {
        this.value = this.editor.getValue()

        this.dispatchEvent(
            new CustomEvent('on-input', {
                bubbles: true,
                composed: true,
                detail: {
                    name: this.name,
                    value: this.value,
                },
            })
        )
    }

    async loadMonaco() {
        if (typeof window.monaco == 'undefined') {
            await this.loadMonacoScript()
        }

        await this.loadMoncoStyles()
    }

    loadMonacoScript() {
        return new Promise((resolve) => {
            window.require = {
                paths: { vs: '/assets/lib/monaco-editor/min/vs' },
            }

            let node

            node = document.createElement('script')

            node.onload = resolve

            node.src = '/assets/lib/monaco-editor/min/vs/loader.js'

            document.body.appendChild(node)
        })
    }

    async loadMoncoStyles() {
        let css

        if (!this.constructor.__monacoStyles) {
            css = await (
                await fetch(
                    '/assets/lib/monaco-editor/min/vs/editor/editor.main.css'
                )
            ).text()

            this.constructor.__monacoStyles = css
        } else {
            css = this.constructor.__monacoStyles
        }

        const style = document.createElement('style')

        style.innerHTML = css

        this.shadowRoot.appendChild(style)
    }

    setLanguage = (language) => {
        window.monaco.editor.setModelLanguage(this.editor.getModel(), language)
    }

    renderErrors() {
        return !isEmpty(this.errors)
            ? html` <label class="error"> ${this.errors[0]} </label> `
            : html``
    }

    render() {
        return html`
            <label>
                <slot name="label"></slot>
            </label>

            ${this.loading ? html`<qrcg-loader></qrcg-loader>` : null}

            <div id="editor-container"></div>

            ${this.renderErrors()}
        `
    }
}
customElements.define('qrcg-code-input', QrcgCodeInput)
