import { html, css, LitElement } from 'lit'
import { classMap } from 'lit/directives/class-map.js'
import { debounce, random } from '../core/helpers'
import { mdiRefresh } from '@mdi/js'

export class QRCGColorPicker extends LitElement {
    static get styles() {
        return css`
            :host {
                display: flex;
                flex-direction: column;
            }

            .control-container {
                display: flex;
                align-items: center;
                flex-wrap: wrap;
            }

            .color-box,
            .input-box {
                width: 2.7rem;
                height: 2rem;
                cursor: pointer;
                margin: 0.25rem;
                position: relative;
                border-radius: 0.25rem;
                outline: 0;
                border: 2px solid transparent;
            }

            .color-box {
                -webkit-tap-highlight-color: transparent;
                user-select: none;
                -webkit-user-select: none;
                touch-action: manipulation;
            }

            .input-box {
                margin: 0;
                overflow: hidden;
                /* border: 2px solid white; */
            }

            input {
                appearance: none;
                -webkit-appearance: none;
                background-color: transparent;
                border: 0;

                margin: 0;
                padding: 0;
                border-radius: 0;
                width: 2.7rem;
                height: 2rem;
                cursor: pointer;
            }

            input::-webkit-color-swatch-wrapper {
                padding: 0 !important;
            }

            input::-webkit-color-swatch {
                border: 0;

                padding: 0;
            }

            .input-wrapper {
                display: flex;
                padding: 0.15rem;
                border-radius: 0.25rem;

                background: radial-gradient(
                        circle at 50% 0,
                        rgba(255, 0, 0, 0.8),
                        rgba(255, 0, 0, 0) 70.71%
                    ),
                    radial-gradient(
                        circle at 6.7% 75%,
                        rgba(0, 0, 255, 0.8),
                        rgba(0, 0, 255, 0) 70.71%
                    ),
                    radial-gradient(
                            circle at 93.3% 75%,
                            rgba(0, 255, 0, 0.8),
                            rgba(0, 255, 0, 0) 70.71%
                        )
                        beige;

                box-sizing: border-box;
            }

            .color-box:nth-child(2) {
                display: none;
            }

            @media (min-width: 800px) {
                .color-box:nth-child(2) {
                    display: block;
                }
            }

            .color-box:first-of-type {
                margin-left: 0;
            }

            .color-box:focus,
            .color-box.selected {
                border: 2px solid black;
            }

            label {
                user-select: none;
                -webkit-user-select: none;
                font-weight: bold;
                margin-right: 1rem;

                font-size: 0.8rem;
            }

            label.empty {
                display: none;
            }

            .refresh-icon {
                cursor: pointer;
                margin-left: 0.25rem;
                touch-action: manipulation;
                -webkit-tap-highlight-color: transparent;
                user-select: none;
            }

            @media (hover) {
                .refresh-icon:hover {
                    animation: rotate 1s linear both infinite;
                }
            }

            @supports (-webkit-touch-callout: none) {
                /* CSS specific to iOS devices */
                input {
                    height: 40px;
                }
            }

            @keyframes rotate {
                from {
                    transfrom: rotate(0deg);
                }

                to {
                    transform: rotate(360deg);
                }
            }
        `
    }

    static get properties() {
        return {
            colors: {
                type: Array,
            },
            value: {},

            name: {},

            labelIsEmpty: {
                state: true,
            },

            maxPalleteShade: {
                type: Number,
                attribute: 'max-pallete-shade',
            },
        }
    }

    static randomColorsCount = 5

    static get presetColors() {
        return this.generateColorPalette(255)
    }

    static generateColorPalette(max) {
        return Array.from({ length: this.randomColorsCount }).map(() => {
            return this.rgbToHex(random(0, max), random(0, max), random(0, max))
        })
    }

    static componentToHex(c) {
        var hex = c.toString(16)
        return hex.length == 1 ? '0' + hex : hex
    }

    static rgbToHex(r, g, b) {
        return (
            '#' +
            this.componentToHex(r) +
            this.componentToHex(g) +
            this.componentToHex(b)
        )
    }

    constructor() {
        super()

        this.maxPalleteShade = 255

        this.colors = this.constructor.generateColorPalette(
            this.maxPalleteShade
        )

        this._debouncedSetValue = debounce(this._debouncedSetValue, 300)
    }

    connectedCallback() {
        super.connectedCallback()

        this.addEventListener('click', this.onClick)

        this.addEventListener('keypress', this._onKeypress)

        this.addEventListener('slotchange', this._checkIfLabelIsEmpty)
    }

    disconnectedCallback() {
        super.disconnectedCallback()

        this.removeEventListener('click', this.onClick)

        this.removeEventListener('keypress', this._onKeypress)

        this.removeEventListener('slotchange', this._checkIfLabelIsEmpty)
    }

    onClick = (e) => {
        if (e.composedPath()[0].matches('.color-box')) {
            this._onBoxClick(e)
        }
    }

    _onKeypress = (e) => {
        if (e.composedPath()[0].matches('.color-box') && e.key === 'Enter') {
            this._onBoxClick(e)
        }
    }

    refreshColors() {
        this.colors = this.constructor.generateColorPalette(
            this.maxPalleteShade
        )
    }

    firstUpdated() {
        this._checkIfLabelIsEmpty()

        this._input().value = this.value
    }

    _checkIfLabelIsEmpty = () => {
        const slot = this.renderRoot.querySelector('slot')

        const text = slot.assignedNodes().reduce((text, el) => {
            return text + el.text
        }, '')

        this.labelIsEmpty = text.trim().length === 0
    }

    _onBoxClick(e) {
        this._setValue(e.composedPath()[0].getAttribute('color'))
    }

    _setValue(value) {
        this.value = value

        this.dispatchEvent(
            new CustomEvent('on-input', {
                bubbles: true,
                composed: true,
                detail: { value: this.value, name: this.name },
            })
        )
    }

    _onCustomColorChange = (e) => {
        this._debouncedSetValue(e.target.value)
    }

    _debouncedSetValue = (value) => {
        this._setValue(value)
    }

    _input() {
        return this.renderRoot.querySelector('input')
    }

    willUpdate(changed) {
        if (changed.has('maxPalleteShade')) {
            this.refreshColors()
        }
    }

    updated(changed) {
        if (changed.has('value')) {
            this._input().value = this.value
        }
    }

    render() {
        return html`
            <label class=${classMap({ empty: this.labelIsEmpty })}
                ><slot></slot
            ></label>

            <div class="control-container">
                ${this.colors.map((c) => {
                    const style = `background-color: ${c};`

                    return html`
                        <div
                            tabindex="0"
                            class="color-box ${classMap({
                                selected: c === this.value,
                            })}"
                            style="${style}"
                            color=${c}
                            part="color-box"
                        ></div>
                    `
                })}

                <div class="input-wrapper">
                    <div class="input-box" part="color-box">
                        <input
                            type="color"
                            @change=${this._onCustomColorChange}
                        />
                    </div>
                </div>
                <qrcg-icon
                    mdi-icon=${mdiRefresh}
                    class="refresh-icon"
                    @click=${this.refreshColors}
                    width="40px"
                    height="25px"
                ></qrcg-icon>
            </div>
        `
    }
}

window.customElements.define('qrcg-color-picker', QRCGColorPicker)
