import { LitElement, html, css } from 'lit'
import { loggedIn, logout } from '../core/auth'
import { mdiMenu } from '@mdi/js'
import './qrcg-drawer'

import '../ui/qrcg-app-logo'

import { homePage } from '../models/user'

import { t } from '../core/translate'

class QRCGHeader extends LitElement {
    static get styles() {
        return css`
            :host {
                display: flex;
                padding: 0.5rem 1rem;
                background-color: var(--header-bg);
                color: white;
                align-items: center;
            }

            .push {
                flex: 1;
            }

            .account-link {
                text-transform: uppercase;
                font-size: 0.8rem;
                letter-spacing: 2px;
                font-weight: bold;
                margin-right: 1rem;
                color: white;
                text-decoration: none;
                cursor: pointer;
                user-select: none;
                -webkit-user-select: none;
            }

            .account-link:hover {
                text-decoration: underline;
            }

            .logout {
                display: none;
            }

            .menu-icon {
                cursor: pointer;
                width: 2rem;
                height: 2rem;
            }

            @media (min-width: 900px) {
                .menu-icon {
                    width: 1rem;
                    height: 1rem;
                }

                .logout {
                    display: block;
                }
            }

            .menu-icon:hover {
                transform: scale(1.3);
            }

            .menu-item:hover {
                text-decoration: underline;
            }

            .menu-item:hover::after {
                color: var(--gray-2);
            }

            qrcg-free-trial-button {
                margin-right: 1rem;
                display: none;
            }

            @media (min-width: 850px) {
                qrcg-free-trial-button {
                    display: flex;
                }
            }

            qrcg-free-trial-button::part(button) {
                font-size: 0.8rem;
                padding: 0.5rem 1rem;
            }

            qrcg-app-logo::part(logo-img) {
                max-height: 3rem;
                max-width: 20rem;
            }
        `
    }

    static get properties() {
        return {
            drawerMenu: { state: true },
            opened: { type: Boolean },
        }
    }

    constructor() {
        super()

        this.drawerMenu = [
            {
                label: t('Why Us'),
                link: '/#why-us',
            },
            {
                label: t('Dynamic vs Static QR Codes'),
                link: '/#dynamic-vs-static-qrcodes',
            },
            {
                label: t('Pricing'),
                link: '/#pricing',
            },
            {
                label: t('FAQs'),
                link: '/#faqs',
            },
        ]

        this.opened = false
    }

    connectedCallback() {
        super.connectedCallback()
    }

    disconnectedCallback() {
        super.disconnectedCallback()
    }

    onLogout(e) {
        e.preventDefault()
        e.stopImmediatePropagation()

        logout()
    }

    renderLoginLink() {
        const home = homePage()

        if (loggedIn()) {
            if (home) {
                return html`
                    <a class="account-link" href="${home}">${t('Dashboard')}</a>
                    <div class="account-link logout" @click=${this.onLogout}>
                        ${t('Logout')}
                    </div>
                `
            } else {
                return html`<a class="account-link" @click=${this.onLogout}>
                    ${t('Logout')}
                </a>`
            }
        }

        return html`<a class="account-link" href="/account/login"
            >${t('Login')}</a
        >`
    }

    openDrawer() {
        this.opened = true
    }

    closeDrawer() {
        this.opened = false
    }

    onMenuItemClick(e) {
        if (e.target.pathname !== window.location.pathname) {
            return
        }
        e.preventDefault()
        e.stopPropagation()

        this.closeDrawer()

        const selector = e.currentTarget.hash

        const elem = document.querySelector(selector)

        elem.scrollIntoView({
            behavior: 'smooth',
        })
    }

    render() {
        return html`
            <qrcg-app-logo href="/"></qrcg-app-logo>
            <div class="empty"></div>
            <div class="push"></div>

            ${!loggedIn()
                ? html`<qrcg-free-trial-button
                      >${t('START FREE TRIAL')}</qrcg-free-trial-button
                  >`
                : ''}
            ${this.renderLoginLink()}

            <qrcg-icon
                class="menu-icon"
                mdi-icon=${mdiMenu}
                @click=${this.openDrawer}
            ></qrcg-icon>

            <qrcg-drawer
                .opened=${this.opened}
                @request-close=${this.closeDrawer}
            >
                ${this.drawerMenu.map(
                    (item) =>
                        html`<a
                            class="menu-item"
                            href=${item.link}
                            @click=${this.onMenuItemClick}
                            >${item.label}</a
                        >`
                )}
            </qrcg-drawer>
        `
    }
}

window.customElements.define('qrcg-header', QRCGHeader)
