import { LitElement, html, css } from 'lit'

import './qrcg-box'

import './qrcg-icon'

import { mdiCloseCircle } from '@mdi/js'

import { parentMatches } from '../core/helpers'

export class QrcgModal extends LitElement {
    static get styles() {
        return css`
            :host {
                display: none;
            }

            :host([opened]) {
                display: flex;
                position: fixed;
                top: 0;
                bottom: 0;
                left: 0;
                right: 0;
                z-index: 100000;
                align-items: center;
                justify-content: center;
                padding: 1rem;
                animation: modal-fade-in ease-in 0.2s both;
            }

            :host([opened][closing]) {
                animation: modal-fade-out ease-out 0.2s both;
            }

            .overlay {
                position: absolute;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background-color: black;
                opacity: 0.8;
            }

            .container {
                position: relative;
                z-index: 1;
                padding: 0;
                width: 100%;
            }

            @media (min-width: 500px) {
                .container {
                    max-width: 450px;
                }
            }

            @media (min-width: 1300px) {
                .container {
                    max-width: 550px;
                }
            }

            .close {
                color: var(--primary-0);
                cursor: pointer;
            }

            @media (min-width: 900px) {
                .close {
                    width: 1.5rem;
                    height: 1.5rem;
                }
            }

            .toolbar {
                display: flex;
                align-items: center;
                padding: 1rem;
                justify-content: space-between;
                border-bottom: 1px solid var(--gray-1);
            }

            h3 {
                margin: 0;
            }

            .body {
                padding: 1rem;
                border-bottom: 1px solid var(--gray-1);
                line-height: 1.7rem;
            }

            .actions {
                display: flex;
                padding: 1rem;
                align-items: center;
                justify-content: flex-end;
            }

            .actions qrcg-button:last-child {
                margin-left: 1rem;
            }

            @keyframes modal-fade-in {
                from {
                    opacity: 0;
                    transform: translateY(1%);
                }

                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }

            @keyframes modal-fade-out {
                from {
                    opacity: 1;
                    transform: translateY(0);
                }

                to {
                    opacity: 0;
                    transform: translateY(1%);
                }
            }
        `
    }

    static get properties() {
        return {
            title: {},
            modalRootId: {},
            closing: { type: Boolean, reflect: true },
            opened: { type: Boolean, reflect: true },
        }
    }

    constructor() {
        super()

        this.close = this.close.bind(this)
        this.onAnimationEnd = this.onAnimationEnd.bind(this)
        this.onNeutralClose = this.onNeutralClose.bind(this)

        this.onKeyup = this.onKeyup.bind(this)
        this.onModalClick = this.onModalClick.bind(this)
    }

    connectedCallback() {
        super.connectedCallback()

        document.addEventListener('keyup', this.onKeyup)

        this.addEventListener('click', this.onModalClick)

        this.addEventListener('animationend', this.onAnimationEnd)
    }

    disconnectedCallback() {
        super.disconnectedCallback()

        document.removeEventListener('keyup', this.onKeyup)

        this.removeEventListener('click', this.onModalClick)

        this.removeEventListener('animationend', this.onAnimationEnd)
    }

    initPromise() {
        this.openPromise = new Promise((resolve, reject) => {
            this.resolve = resolve
            this.reject = reject
        })

        return this.openPromise
    }

    async open() {
        this.opened = true

        return this.initPromise()
    }

    onOverlayClick() {
        this.onNeutralClose()
    }

    close() {
        this.closing = true
    }

    onAnimationEnd() {
        if (this.closing) {
            this.opened = false

            this.closing = false

            this.remove()
        }
    }

    onKeyup(e) {
        if (e.key === 'Escape') {
            this.close()
        }
    }

    onModalClick(e) {
        let node = e.composedPath()[0]

        if (parentMatches(node, '[modal-affirmative')) {
            this.onAffirmativeClick()
        }

        if (parentMatches(node, '[modal-negative]')) {
            this.onNegativeClick()
        }

        if (parentMatches(node, '.close, .overlay')) {
            this.onNeutralClose()
        }
    }

    onNeutralClose() {
        this.reject()

        this.close()
    }

    onAffirmativeClick() {
        this.resolve()

        this.close()
    }

    onNegativeClick() {
        this.reject()

        this.close()
    }

    renderBody() {}

    renderActions() {}

    renderTitle() {}

    render() {
        return html`
            <div class="overlay"></div>

            <qrcg-box class="container">
                <div class="toolbar">
                    <h3>${this.renderTitle()}</h3>
                    <qrcg-icon
                        mdi-icon=${mdiCloseCircle}
                        class="close"
                    ></qrcg-icon>
                </div>

                <div class="body">${this.renderBody()}</div>

                <div class="actions">${this.renderActions()}</div>
            </qrcg-box>
        `
    }
}
