import { LitElement, html, css } from 'lit'

import { push } from '../core/qrcg-router'

export class QrcgTab extends LitElement {
    static styles = [
        css`
            :host {
                display: inline-block;
                margin-right: 0.5rem;
                margin: 1rem 0.5rem 1rem 0.1rem;
            }

            qrcg-button {
                display: inline-block;
            }

            qrcg-button::part(button) {
                background-color: var(--gray-0);
                color: black;
            }

            :host([active]) qrcg-button::part(button) {
                background-color: var(--gray-1);
                color: black;
            }
        `,
    ]

    static get properties() {
        return {
            tabId: {
                attribute: 'tab-id',
            },
            active: {
                type: Boolean,
                reflect: true,
            },
        }
    }

    connectedCallback() {
        super.connectedCallback()

        document.addEventListener('qrcg-tab:activate', this.onActivate)
    }

    disconnectedCallback() {
        super.disconnectedCallback()

        document.removeEventListener('qrcg-tab:activate', this.onActivate)
    }

    firstUpdated() {
        if (this.active && !this.getQueryParam()) {
            this.fireActivate()
        } else {
            this.loadQueryParam()
        }
    }

    getQueryParam() {
        return new URLSearchParams(location.search).get('tab-id')
    }

    loadQueryParam() {
        if (this.getQueryParam() == this.tabId) {
            this.fireActivate()
        }
    }

    onActivate = (e) => {
        if (e.detail.tabId === this.tabId) {
            this.active = true
        } else {
            this.active = false
        }

        this.syncQueryParam()
    }

    onLabelClick() {
        this.fireActivate()
    }

    fireActivate() {
        document.dispatchEvent(
            new CustomEvent('qrcg-tab:activate', {
                detail: {
                    tabId: this.tabId,
                },
            })
        )
    }

    syncQueryParam() {
        const s = new URLSearchParams(location.search)

        const tabId = s.get('tab-id')

        if (!this.firstRender && this.active && tabId !== this.tabId) {
            s.set('tab-id', this.tabId)

            s.toString()

            const newLocation = `${location.pathname}?${s.toString()}`

            push(newLocation)
        }

        this.firstRender = false
    }

    render() {
        return html`
            <qrcg-button @click=${this.onLabelClick}>
                <slot></slot>
            </qrcg-button>
        `
    }
}

customElements.define('qrcg-tab', QrcgTab)
