import { LitElement, html, css } from 'lit'
import { isEmpty, isFunction } from '../core/helpers'

import { LoadableTable } from './loadable-table'

export class QrcgTable extends LoadableTable(LitElement) {
    static styles = [
        ...super.styles,

        css`
            :host {
                display: block;
            }

            table {
                border-collapse: collapse;
                margin: 0;
                padding: 0;
                width: 100%;
                table-layout: fixed;
            }

            table tr {
                padding: 0.35em;
            }

            table thead {
                background-color: var(--gray-0);
            }

            table td {
                padding: 1rem 0.5rem;
                text-align: center;
                border-bottom: 1px solid var(--gray-1);
                color: var(--dark);
                text-overflow: ellipsis;
                overflow: hidden;
                line-height: 1.6;
            }

            table th {
                padding: 1rem 0.5rem;
                text-align: center;
                color: var(--gray-2);
                user-select: none;
                -webkit-user-select: none;
            }

            table th {
                font-size: 0.85em;
                letter-spacing: 0.1em;
                text-transform: uppercase;
            }

            .column-actions a {
                color: var(--primary-0);
                font-size: 0.8rem;
                display: inline-block;
                margin: 0.25rem;
                text-decoration: underline;
            }

            td a {
                color: var(--primary-0);
                cursor: pointer;
                touch-action: manipulation;
                user-select: none;
            }

            @media screen and (max-width: 600px) {
                table {
                    border: 0;
                }

                table thead {
                    border: none;
                    clip: rect(0 0 0 0);
                    height: 1px;
                    margin: -1px;
                    overflow: hidden;
                    padding: 0;
                    position: absolute;
                    width: 1px;
                }

                table tr {
                    border-bottom: 3px solid #ddd;
                    display: block;
                    margin-bottom: 0.625em;
                }

                table td {
                    border-bottom: 1px solid #ddd;
                    display: flex;
                    font-size: 0.8em;
                    text-align: right;
                    justify-content: space-between;
                }

                table td::before {
                    /*
    * aria-label has no advantage, it won't be read inside a table
    content: attr(aria-label);
    */
                    content: attr(data-label);
                    font-weight: bold;
                    text-transform: uppercase;
                }

                table td:last-child {
                    border-bottom: 0;
                }
            }
        `,
    ]

    static get properties() {
        return {
            ...super.properties,
            columns: {},
            rows: {},
            loading: { type: Boolean, reflect: true },
            noResultsMessage: {
                attribute: 'no-results-message',
            },
        }
    }

    constructor() {
        super()

        this.loading = false
        this.columns = []
        this.rows = []

        this.renderCell = this.renderCell.bind(this)
        this.renderCellContent = this.renderCellContent.bind(this)

        this.noResultsMessage = 'No results could be found'
    }

    columnStyles() {
        const styles = this.columns
            .map((column) => {
                if (column.width) {
                    return `
            @media (min-width: 601px) {
                .column-${column.key} { width: ${column.width}}
            }
        `
                }
                return ''
            })
            .join('')

        return html`<style>
            ${styles}
        </style>`
    }

    renderHeader() {
        return this.columns.map(
            (column) =>
                html`
                    <th scope="col" class="column-${column.key}">
                        ${column.label}
                    </th>
                `
        )
    }

    renderCell(row, column) {
        if (isFunction(this.cellRenderer)) {
            return this.cellRenderer(row, column)
        }

        return html`
            <td data-label="${column.label}" class="column-${column.key}">
                ${this.renderCellContent(row, column)}
            </td>
        `
    }

    renderCellContent(row, column) {
        if (isFunction(this.cellContentRenderer)) {
            return this.cellContentRenderer(row, column)
        }

        return row[column.key]
    }

    setCellRenderer(cp) {
        this.cellRenderer = cp
    }

    setCellContentRenderer(cp) {
        this.cellContentRenderer = cp
    }

    renderRow(row) {
        return html`
            <tr>
                ${this.columns.map((column) => this.renderCell(row, column))}
            </tr>
        `
    }

    renderNoRowsAvailableMessage() {
        return html`
            <tr>
                <td colspan="${this.columns.length}">
                    ${this.noResultsMessage}
                </td>
            </tr>
        `
    }

    renderRows() {
        if (!this.firtsLoad && isEmpty(this.rows)) {
            return this.renderNoRowsAvailableMessage()
        }

        return this.rows.map((row) => this.renderRow(row))
    }

    willUpdate(changed) {
        super.willUpdate?.(changed)
    }

    render() {
        return html`
            ${this.columnStyles()}

            <table>
                <thead>
                    <tr>
                        ${this.renderHeader()}
                    </tr>
                </thead>
                <tbody>
                    ${this.renderRows()}
                </tbody>
            </table>
        `
    }
}
customElements.define('qrcg-table', QrcgTable)
