<?php

namespace App\Http\Controllers;

use App\Interfaces\CurrencyManager;
use App\Models\StripePaymentGatewayModel;
use App\Models\Transaction;
use App\Repositories\StripePaymentGateway;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Stripe\Stripe;
use Stripe\Event as StripeEvent;

use UnexpectedValueException;

class StripeWebhook extends Controller
{
    /**
     * Handle the incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function __invoke(Request $request)
    {
        Stripe::setApiKey(config('services.stripe.secret_key'));

        $payload = $request->getContent();

        $event = null;

        try {
            $event = StripeEvent::constructFrom(
                json_decode($payload, true)
            );
        } catch (UnexpectedValueException $e) {
            return response('⚠️  Webhook error while parsing basic request.', 400);
        }

        $stripe = new StripePaymentGateway(app(CurrencyManager::class));

        // Handle the event
        switch ($event->type) {
            case 'payment_intent.succeeded':
                $paymentIntent = $event->data->object;

                $checkoutSession = $stripe->getCheckoutSessionByPaymentIntent($paymentIntent->id);

                if (!$checkoutSession) {
                    return;
                }

                $transaction = new Transaction();
                $transaction->subscription_id = $checkoutSession->metadata->subscription_id;
                $transaction->amount = $paymentIntent->amount / 100;
                $transaction->currency = $paymentIntent->currency;
                $transaction->payment_gateway_id = StripePaymentGatewayModel::instance()->id;
                $transaction->status = Transaction::STATUS_SUCESS;
                $transaction->stripe_payment_intent_id = $paymentIntent->id;
                $transaction->save();

                break;
            case 'payment_intent.payment_failed':

                $paymentIntent = $event->data->object;

                $checkoutSession = $stripe->getCheckoutSessionByPaymentIntent($paymentIntent->id);

                $transaction = new Transaction();
                $transaction->subscription_id = $checkoutSession->metadata->subscription_id;
                $transaction->amount = $paymentIntent->amount / 100;
                $transaction->currency = $paymentIntent->currency;
                $transaction->payment_gateway_id = StripePaymentGatewayModel::instance()->id;
                $transaction->status = Transaction::STATUS_FAILED;
                $transaction->stripe_payment_intent_id = $paymentIntent->id;
                $transaction->save();

                break;
            default:
                Log::error(static::class . ' Received unknown event type');
        }

        return 'OK';
    }
}
