<?php

namespace App\Http\Controllers;

use App\Interfaces\FileManager;
use App\Models\Config;
use App\Support\SoftwareUpdate\DatabaseUpdateManager;
use App\Support\SystemStatus\SystemStatus;
use DateTimeZone;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;

class SystemController extends Controller
{
    private DatabaseUpdateManager $dbUpdateManager;

    public function __construct(DatabaseUpdateManager $dbUpdateManager)
    {
        $this->dbUpdateManager = $dbUpdateManager;
    }

    public function status()
    {
        $status = new SystemStatus();

        return [
            'ok' => $status->ok() && !$this->dbUpdateManager->hasDatabaseUpdate(),
            'entries' => $status->get()
        ];
    }

    public function checkDatabaseUpdate()
    {
        return [
            'update_available' => $this->dbUpdateManager->hasDatabaseUpdate()
        ];
    }

    public function updateDatabase()
    {
        $this->dbUpdateManager->updateDatabase();

        return [
            'updated' => true
        ];
    }

    public function getTimezones()
    {
        return DateTimeZone::listIdentifiers();
    }

    public function cron()
    {
        Artisan::call('schedule:run');

        Artisan::call('queue:work --stop-when-empty');
    }

    public function saveConfigs(Request $request)
    {
        $this->restrictDemo();

        if (!$request->user()->permitted('system.settings')) {
            abort(403);
        }

        foreach ($request->all() as $input) {
            Config::set($input['key'], @$input['value']);
        }

        return array_map(function ($input) {
            return [
                'key' => $input['key'],
                'value' => Config::get($input['key'])
            ];
        }, $request->all());
    }

    public function getConfigs(Request $request)
    {
        if (!$request->user()->permitted('system.settings')) {
            abort(403);
        }

        $keys = explode(',', $request->keys);

        $configsArray = array_map(function ($key) {
            return [
                'key' => $key,
                'value' => Config::get($key) ?? config($key)
            ];
        }, $keys);

        $configsArray = $this->hideInDemo($configsArray, [
            'services.google.api_key'
        ]);

        return $configsArray;
    }

    public function uploadConfigAttachment(Request $request, FileManager $files)
    {
        $this->restrictDemo();

        $key = $request->key;

        if (!$request->user()->permitted('system.settings')) {
            abort(403);
        }

        if (!Config::id($key)) {
            Config::set($key, null);
        }

        $request->merge([
            'attachable_type' => Config::class,
            'attachable_id' => Config::id($key),
            'type' => FileManager::FILE_TYPE_CONFIG_ATTACHMENT
        ]);

        $file = $files->store($request);

        Config::set($key, $file->id);

        return $file;
    }

    private function hideInDemo($configsArray, $keysToHide)
    {
        if (!app()->environment('demo')) {
            return $configsArray;
        }

        return array_map(function ($config) use ($keysToHide) {
            $shouldHide = array_filter($keysToHide, fn ($key) => $key === $config['key']);

            if ($shouldHide) {
                return array_merge(
                    $config,
                    ['value' => 'hidden_in_demo']
                );
            }

            return $config;
        }, $configsArray);
    }
}
