<?php

namespace App\Http\Controllers;

use App\Http\Middleware\ErrorMessageMiddleware;
use Illuminate\Http\Request;

use App\Models\User;
use App\Http\Requests\StoreUserRequest;
use App\Http\Requests\UpdateUserRequest;
use App\Interfaces\ModelIndex;
use App\Interfaces\UserManager;
use App\Models\Role;
use App\Models\Subscription;
use App\Models\SubscriptionStatus;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;


class UsersController extends Controller
{
    private UserManager $users;

    public function __construct(UserManager $users)
    {
        $this->users = $users;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request, ModelIndex $index)
    {
        if ($request->list_all) {
            return User::all();
        }

        return $index->build(new User, $request)
            ->withQuery(function ($query) use ($request) {
                $query->with('transactions');

                if ($request->has('paying')) {

                    $paying = $request->boolean('paying');

                    if ($paying) {
                        $query->has('transactions');
                    } else {
                        $query->doesntHave('transactions');
                    }
                }
            })
            ->list(['name', 'email']);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreUserRequest $request)
    {
        $user = User::create(array_merge($request->all(), [
            'password' => Hash::make($request->password)
        ]));

        $user->markEmailAsVerified();

        $this->users->changeRole($user, Role::find($request->role_id));

        return $user;
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(User $user)
    {
        return $user;
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateUserRequest $request, User $user)
    {
        $shouldChangeRole = !empty($request->role_id) && $request->role_id != $user->roles->first()?->id;

        if ($shouldChangeRole) {
            $this->authorizeChangeRole($request, $user);
        }

        $user->fill($request->all());

        if ($request->password) {
            $user->password = Hash::make($request->password);
        }

        $anotherUser = User::whereEmail($user->email)->first();

        if ($anotherUser) {
            $emailExists = $anotherUser->id !== $user->id;

            if ($emailExists) {
                throw ValidationException::withMessages([
                    'email' => ['This email is taken.'],
                ]);
            }
        }

        $user->save();

        if ($shouldChangeRole) {
            $this->users->changeRole($user, Role::findOrFail($request->role_id));
        }

        $user->refresh();

        return $user;
    }

    private function authorizeChangeRole(Request $request, User $user)
    {
        if (!$request->user()->permitted('user.update-any')) {
            ErrorMessageMiddleware::setMessage(t('This action is not authorized'));
            abort(403);
        }

        if ($user->id === $request->user()->id) {
            ErrorMessageMiddleware::setMessage(t('Cannot change role of logged in user.'));
            abort(422);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(User $user)
    {
        $this->users->deleteUser($user);

        return $user;
    }
}
