<?php

namespace App\Models;

use App\Events\QRCodeSaved;
use App\Repositories\FileManager;
use App\Support\QRCodeProcessors\AdvancedShapeProcessors\BaseAdvancedShapeProcessor;
use App\Support\QRCodeProcessors\AdvancedShapeProcessors\CouponAdvancedShape;
use App\Support\QRCodeProcessors\GradientProcessor;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;

use Illuminate\Database\Eloquent\Model;

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;


class QRCode extends Model
{
    use HasFactory;

    protected $fillable = ['name', 'type', 'design', 'data'];

    protected $with = ['logo', 'foreground_image'];

    protected $appends = ['logo_id', 'foreground_image_id'];

    protected $table = 'qrcodes';

    protected $casts = [
        'data' => 'object',
        'design' => 'object',
        'archived' => 'boolean',
    ];

    protected $dispatchesEvents = [
        'saved' => QRCodeSaved::class,
    ];

    private $white = '#ffffff';

    private $primaryColor = '#1c57cb';

    private $black = '#000000';

    public static function getTypes()
    {
        return [
            'url',
            'vcard',
            'text',
            'email',
            'sms',
            'wifi',
            'whatsapp',
            'facetime',
            'location',
            'event',
            'crypto',
            'paypal',
            'zoom',
            'skype',
            'brazilpix',
            'call',
            'telegram',
            'facebookmessenger',
            'wechat',
            'viber',
        ];
    }

    protected static function booted()
    {
        static::creating([static::class, 'onCreating']);

        static::saving([static::class, 'onSaving']);
    }


    static function onCreating($model)
    {
        $model->generateFileName();
    }

    static function onSaving($model)
    {
        $model->generateNameIfNeeded();
    }

    private function generateNameIfNeeded()
    {
        if (!empty($this->name)) {
            return;
        }

        switch ($this->type) {
            case 'url':
                if (isset($this->data->url))
                    $this->name = Str::limit($this->data->url, 20);
                else
                    $this->name = 'URL QR Code';
                break;
            case 'text':
                if (isset($this->data->text))
                    $this->name = Str::words($this->data->text, 3, '...');
                else
                    $this->name = 'Plain text';
                break;
            case 'vcard':
                if (isset($this->data->firstName))
                    $this->name = 'VCARD of ' . $this->data->firstName;
                else
                    $this->name = 'VCARD';
                break;
            case 'wifi':
                if (isset($this->data->ssid))
                    $this->name = 'WIFI ' . $this->data->ssid;
                else
                    $this->name = 'WIFI';
                break;
            case 'email':
                if (isset($this->data->email))
                    $this->name = 'EMAIL to ' . $this->data->email;
                else
                    $this->name = 'Email';
                break;
            case 'sms':
                if (isset($this->data->message))
                    $this->name = 'SMS ' . Str::words($this->data->message, 3, '...');
                else
                    $this->name = 'SMS';
                break;
            case 'call':
                $this->name = empty($this->data->phone) ? 'CALL' : 'Call to ' . $this->data->phone;
                break;
            case 'event':
                $this->name = empty($this->data->event_name) ? 'EVENT' : 'Event: ' . $this->data->event_name;
                break;
            case 'whatsapp':
                $this->name = empty($this->data->mobile_number) ? 'WHATSAPP' : 'WhatsApp to: ' . $this->data->mobile_number;
                break;
            case 'facetime':
                $this->name = empty($this->data->target) ? 'FACETIME' : 'Facetime to: ' . $this->data->target;
                break;
            default:
                $this->name = strtoupper($this->type);
                break;
        }
    }


    public function generateFileName()
    {
        $fileName = '';

        $found = true;

        do {
            $fileName =  uniqid();

            $found = QRCode::where('file_name', $fileName)->first();
        } while ($found);

        $this->file_name = $fileName;
    }

    public function redirect()
    {
        return $this->hasOne(QRCodeRedirect::class, 'qrcode_id');
    }




    public function setDesignAttribute($designParam)
    {
        $designParam = (object) $designParam;

        $defaultDesign = array_merge(
            [
                'fillType' => 'solid',
                'foregroundColor' => $this->black,
                'eyeInternalColor' => $this->black,
                'eyeExternalColor' => $this->black,
                'gradientFill' => GradientProcessor::defaultGradient(),
                'module' => 'square',
                'shape' => 'none',
                'frameColor' => $this->black,
                'logoScale' => 0.2, // default
                'logoPositionX' => 0.5,
                'logoPositionY' => 0.5,
                'logoRotate' => 0,
                'logoBackground' => true,
                'logoBackgroundFill' => $this->white,
                'logoUrl' => null,
                'logoType' => 'preset',
                'logoBackgroundScale' => 1.5,
                'logoBackgroundShape' => 'circle',
                'backgroundEnabled' => true,
                'backgroundColor' => $this->white,
                'finder' => 'default',
                'finderDot' => 'default',
                'advancedShape' => 'none',
                'advancedShapeDropShadow' => true,
                'advancedShapeFrameColor' => $this->black,

            ],
            BaseAdvancedShapeProcessor::defaultTextRelatedValues(),
            CouponAdvancedShape::defaultValues(),
        );

        $design = (object) [];

        if (!$designParam) {
            $designParam = (object) [];
        }

        foreach ($defaultDesign as $key => $defaultValue) {
            $design->$key = !isset($designParam->$key) ? $defaultValue : $designParam->$key;
        }

        $this->attributes['design'] = json_encode($design);
    }

    public function logo()
    {
        return $this
            ->morphOne(File::class, 'attachable')
            ->where(
                'type',
                FileManager::FILE_TYPE_QRCODE_LOGO
            );
    }

    public function foreground_image()
    {
        return $this
            ->morphOne(File::class, 'attachable')
            ->where(
                'type',
                FileManager::FILE_TYPE_QRCODE_FOREGROUND_IMAGE
            );
    }

    public function foregroundImageId(): Attribute
    {
        return new Attribute(fn () => $this->foreground_image?->id);
    }

    public function logoId(): Attribute
    {
        return new Attribute(fn () => $this->logo?->id);
    }

    public function getFilePath($extension)
    {
        return Storage::path(config('qrcode.storage_path') . '/' . $this->file_name . '.' . $extension);
    }


    public function user()
    {
        return $this->belongsTo(User::class);
    }
}
