<?php

namespace App\Models;

use App\Events\UserCreated;
use App\Jobs\SendVerificationEmail;
use App\Notifications\QueuedVerifyEmail;
use Exception;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Log;
use Laravel\Sanctum\HasApiTokens;

use Illuminate\Support\Str;


/**
 * @property \Illuminate\Database\Eloquent\Collection subscriptions
 * @property \Illuminate\Database\Eloquent\Collection roles
 * @property \Illuminate\Database\Eloquent\Collection transactions
 * @property \Illuminate\Database\Eloquent\Collection qrcodes
 * @property \Illuminate\Database\Eloquent\Collection permissions
 */
class User extends Authenticatable implements MustVerifyEmail
{
    use HasApiTokens, HasFactory, Notifiable;

    static $count = 0;


    protected $with = ['roles', 'roles.permissions'];

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
    ];

    protected $dispatchesEvents = [
        'created' => UserCreated::class,
    ];

    public function isSuperAdmin()
    {
        $superAdmin = $this->roles->reduce(
            fn ($superAdmin, $role) => $superAdmin || $role->super_admin,
            false
        );

        if ($superAdmin) return true;
    }

    /**
     * Determines if the user has a permission
     * 
     * @param string $slug Permission slug
     * @return boolean
     */
    public function permitted($slug)
    {
        $model = explode('.', $slug)[0];

        $capability = explode('.', $slug)[1];

        if (!preg_match("/-/", $model)) {
            $model = Str::kebab($model);

            $slug = "$model.$capability";
        }

        if ($this->isSuperAdmin()) return true;

        return $this->permissions->first(function ($permission) use ($slug) {
            return $permission->slug === $slug;
        }) ? true : false;
    }

    public function sendEmailVerificationNotification()
    {
        if (config('app.email_verification_after_sign_up'))
            $this->notify(new QueuedVerifyEmail);
    }

    public function permissions(): Attribute
    {
        $permissions = $this
            ->roles
            ->pluck('permissions')
            ->flatten()
            ->unique('id');

        return new Attribute(
            fn () => $permissions
        );
    }

    public function subscriptions()
    {
        return $this->hasMany(Subscription::class)->orderBy('id', 'desc');
    }

    public function roles()
    {
        return $this->belongsToMany(Role::class, 'user_roles');
    }

    public function transactions()
    {
        return $this->hasManyThrough(Transaction::class, Subscription::class);
    }

    public function qrcodes()
    {
        return $this->hasMany(QRCode::class);
    }
}
