<?php

namespace App\Repositories;

use App\Interfaces\FileManager as FileManagerInterface;
use App\Models\File;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator as ValidatorFacade;
use Illuminate\Validation\Validator;
use Throwable;

class FileManager implements FileManagerInterface
{
    private $rejectedExtensions = 'bash|sh|php';

    private $fileValidator;

    public function store(Request $request)
    {
        $this->validate($request);

        $file = $this->save(
            name: $request->file('file')->getClientOriginalName(),
            type: $request->type,
            mime_type: $request->file('file')->getMimeType(),
            attachable_type: $request->attachable_type,
            attachable_id: $request->attachable_id,
            user_id: $request->user()->id,
        );

        $path = $request->file('file')->storeAs(
            static::UPLOAD_DIR,
            $this->makeFileName(
                $file,
                $request->file('file')->extension()
            )
        );

        $file->path = $path;

        $file->save();

        $this->verifyFile($file);

        return $file;
    }

    private function verifyFile($file)
    {
        if (is_callable($this->fileValidator)) {

            try {

                call_user_func($this->fileValidator, $file);
            } catch (Throwable $th) {

                $validator = ValidatorFacade::make(compact('file'), [
                    'file' => 'required'
                ]);

                $validator->after(function ($validator) use ($th) {

                    $validator->errors()->add(
                        'file',
                        $th->getMessage()
                    );
                });

                $this->delete($file);

                $validator->validate();
            }
        }
    }

    private function makeFilePath(File $file, string $extension)
    {
        return $this::UPLOAD_DIR . '/' . $this->makeFileName($file, $extension);
    }

    private function makeFileName(File $file, $extension)
    {
        return $file->id . '.' . $extension;
    }

    public function save(
        $name,
        $type,
        $mime_type,
        $attachable_type,
        $attachable_id,
        $user_id,
        $extension = null,
        $data = null,
    ) {
        $file = new File(
            compact(
                'name',
                'type',
                'mime_type',
                'attachable_type',
                'attachable_id',
                'user_id'
            )
        );

        $file->save();

        if (!empty($data)) {
            $path = $this->makeFilePath($file, $extension);

            Storage::put($path, $data);

            $file->path = $path;

            $file->save();
        }

        return $file;
    }

    public function resource(Request $request, $file)
    {
        $fs = Storage::getDriver();

        $stream = $fs->readStream($file->path);

        return response()->stream(
            function () use ($stream) {
                while (ob_get_level() > 0) ob_end_flush();
                fpassthru($stream);
            },
            200,
            [
                'Content-Type' => 'application/octet-stream',

                'Content-disposition' => sprintf('inline; filename="%s"', $file->name),

                'Content-Length' => filesize($this->path($file))
            ]
        );
    }

    public function validate(Request $request)
    {
        $data = array_merge($request->all(), ['file' => $request->file('file')]);

        $validator = ValidatorFacade::make($data, [
            'file' => 'file|required',
            'attachable_type' => 'required'
        ]);

        $this->validateExtension($request, $validator);

        $this->validateAttachableType($request, $validator);

        $validator->validate();
    }

    public function setFileValidator(Closure $callback)
    {
        $this->fileValidator = $callback;
    }

    private function validateExtension(Request $request, Validator $validator)
    {
        if (empty($request->file('file'))) {
            return;
        }

        $extension = $request->file('file')->clientExtension();

        $rejected = preg_match('/' . $this->rejectedExtensions . '/', $extension);

        $validator->after(function ($validator) use ($rejected) {

            if ($rejected) {
                $validator->errors()->add(
                    'file',
                    'File extension is not supported!'
                );
            }
        });
    }

    private function validateAttachableType(Request $request, Validator $validator)
    {
        $validator->after(function ($validator) use ($request) {

            if (!class_exists($request->attachable_type)) {
                $validator->errors()->add(
                    'file',
                    'Invalid attachable type'
                );
            }
        });
    }

    public function delete(File $file)
    {
        if ($file->path)
            Storage::delete($file->path);

        $file->delete();
    }

    public function path(File $file)
    {
        return Storage::path($file->path);
    }

    public function url(File $file)
    {
        return url(sprintf('/api/files/%s/resource', $file->slug));
    }

    public function copy(File $file, $to)
    {
        copy($this->path($file), $to);
    }

    public function extension(File $file)
    {
        return pathinfo($this->path($file), PATHINFO_EXTENSION);
    }

    public function duplicate(File $file): File
    {
        $copy = $this->save(
            $file->name,
            $file->type,
            $file->mime_type,
            $file->attachable_type,
            $file->attachable_id,
            $file->user_id,
            $this->extension($file)
        );

        $copy->path = $this->makeFilePath($copy, $this->extension($file));

        $this->copy(
            $file,
            $this->path($copy)
        );

        $copy->save();

        return $copy;
    }
}
