<?php

namespace App\Repositories;

use App\Interfaces\QRCodeGenerator as IQRCodeGenerator;
use App\Models\QRCode;
use App\Support\QRCodeDataMakers\BaseMaker as BaseQRCodeDataMaker;
use App\Support\QRCodeOutput;
use Illuminate\Http\Request;
use Illuminate\Pipeline\Pipeline;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class QRCodeGenerator implements IQRCodeGenerator
{
    protected $type;

    protected $data;

    protected $design;

    protected $outputType;

    protected $size;

    protected $model;

    protected static $processors = [];

    public function init(QRCode $model, string $outputType)
    {
        $this->model = $model;
        $this->outputType = $outputType;
        $this->size = config('qrcode.preview_size');
    }

    public function initFromRequest(Request $request)
    {
        if (!empty($request->id)) {
            $model = QRCode::find($request->id);

            if ($model) {
                $this->model = $model;
            }
        }

        if (empty($this->model)) {
            $this->model = new QRCode;
        }

        $this->model->type = $request->type;

        $this->model->data = json_decode(urldecode($request->data));

        $this->model->design = json_decode(urldecode($request->design));

        $this->outputType = $request->outputType;

        $this->size = $request->size ?: config('qrcode.preview_size');
    }


    public function writeString()
    {
        $output = $this->pipe();

        return $output;
    }

    public static function processor($processor)
    {
        static::$processors[] = $processor;
    }

    public function saveVariants(QRCode $model)
    {
        // png will be saved in browser.
        foreach (['svg'] as $variant) {

            $generator = new static;

            $generator->init($model, $variant);

            $path = config('qrcode.storage_path') . "/{$model->file_name}." . $variant;

            Storage::put(
                $path,
                $generator->writeString()
            );
        }
    }

    public function respondInline()
    {
        $this->outputType = 'svg';

        $content = $this->writeString();

        $response = response($content)
            ->header(
                'Content-Type',
                $this->makeContentType()
            );

        return $response;
    }

    protected function pipe()
    {
        $data = BaseQRCodeDataMaker::instance($this->model)->make();

        $output = new QRCodeOutput($this->model, '', $this->size, $data);

        $pipeline = new Pipeline(app());

        $pipeline->send($output)->through(static::$processors);

        $output = $pipeline->thenReturn();

        $output->round = 1;

        $pipeline->send($output)->through($this::$processors);

        $output = $pipeline->thenReturn();

        return $output;
    }

    protected function makeContentType()
    {
        $contentType = 'image/png';

        if (!empty($this->outputType)) {
            return [
                'svg' => 'image/svg+xml',
                'png' => 'image/png'
            ][$this->outputType];
        }

        return $contentType;
    }
}
