<?php

namespace App\Repositories;

use App\Interfaces\UserManager as UserManagerInterface;

use App\Models\QRCodeRedirect;
use App\Models\QRCodeScan;
use App\Models\Role;
use App\Models\SubscriptionStatus;
use App\Models\User;

use Illuminate\Support\Facades\DB;

class UserManager implements UserManagerInterface
{
    public function getSuperAdmins()
    {
        return User::with('roles')->whereHas('roles', function ($role) {
            $role->where('super_admin', true);
        })->get();
    }

    public function getCurrentSubscription(User $user)
    {
        return $user
            ->subscriptions
            ->first(
                fn ($s) => $s->statuses[0]->status
                    !== SubscriptionStatus::STATUS_PENDING_PAYMENT
            );
    }

    public function getCurrentPlan(User $user)
    {
        return $this->getCurrentSubscription($user)->subscription_plan;
    }

    public function deleteUser(User $user)
    {
        $user = User::with('transactions', 'subscriptions', 'qrcodes', 'qrcodes.redirect')->find($user->id);

        $user->transactions->each(function ($transaction) {
            $transaction->delete();
        });

        DB::delete('delete from user_roles where user_id = ?', [$user->id]);

        $user->subscriptions->each(function ($subscription) {
            $subscription->delete();
        });

        $qrcodeIds = $user->qrcodes?->pluck('id');

        $redirectIds = QRCodeRedirect::whereIn('qrcode_id', $qrcodeIds)->pluck('id');

        $scanIds = QRCodeScan::whereIn('qrcode_redirect_id', $redirectIds)->pluck('id');

        DB::table('qrcode_scans')->whereIn('id', $scanIds)->delete();

        DB::table('qrcode_redirects')->whereIn('id', $redirectIds)->delete();

        DB::table('qrcodes')->whereIn('id', $qrcodeIds)->delete();

        $user->delete();
    }

    public function changeRole(User $user, Role $role)
    {
        if (empty($user) || empty($role)) return;

        $relation_record = DB::table('user_roles')->where('user_id', $user->id)->first();

        $data = [
            'user_id' => $user->id,
            'role_id' => $role->id
        ];

        if (!$relation_record) {
            DB::table('user_roles')->insert($data);
        } else {
            DB::table('user_roles')->where('user_id', $user->id)->update($data);
        }
    }
}
