<?php

namespace App\Support\SoftwareUpdate\MandatorySeeders;

use App\Interfaces\FileManager;
use App\Models\Translation;

class TranslationSeeder extends Seeder
{
    protected $rawFile = 'translations';

    protected $table = 'translations';

    private FileManager $files;

    public function __construct(FileManager $files)
    {
        $this->files = $files;
    }

    protected function shouldInsertRow(array $row)
    {
        return Translation::where('locale', $row['locale'])->count() == 0;
    }

    protected function afterRun()
    {
        // Update default translation file on every software update
        $this->updateDefaultTranslationFile();

        $this->saveTranslationFilesIfNeeded();
    }

    protected function saveTranslationFilesIfNeeded()
    {
        foreach (Translation::all() as $translation) {

            $content = @$this->rawFile("translation-$translation->locale.json");

            if (empty($translation->file) && !empty($content)) {

                $this->saveTranslationFile($content, $translation->locale);
            }
        }
    }

    protected function updateDefaultTranslationFile()
    {
        $content = $this->rawFile('translation-en.json');

        $this->saveTranslationFile($content, 'en', true);
    }

    protected function saveTranslationFile($content, $locale, $replace = false)
    {
        $model = Translation::where('locale', $locale)->first();

        if (!$model) return;

        if ($replace) {
            if ($model->file)
                $this->files->delete($model->file);
        }

        $this->files->save(
            name: "translation-{$model->locale}.json",
            type: FileManager::FILE_TYPE_TRANSLATION,
            mime_type: 'text/plain;charset=UTF-8',
            attachable_type: Translation::class,
            attachable_id: $model->id,
            user_id: 1,
            extension: 'json',
            data: $content
        );
    }

    protected function newModel()
    {
        return new Translation;
    }
}
